package notice_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/event/subscription/notice"
	"code.justin.tv/cb/sauron/internal/mocks"

	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB     *mocks.Database
	Pubsub       *mocks.Publisher
	Users        *mocks.Users
	Statsd       *mocks.StatSender
	Zuma         *mocks.Zuma
	AlertManager *mocks.Manager
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:     &mocks.Database{},
		Pubsub:       &mocks.Publisher{},
		Users:        &mocks.Users{},
		Statsd:       &mocks.StatSender{},
		Zuma:         &mocks.Zuma{},
		AlertManager: &mocks.Manager{},
	}
}

func mockHandler(params mockHandlerParams) *notice.Handler {
	return &notice.Handler{
		DynamoDB:     params.DynamoDB,
		Pubsub:       params.Pubsub,
		Users:        params.Users,
		Statsd:       params.Statsd,
		Zuma:         params.Zuma,
		AlertManager: params.AlertManager,
	}
}

func getInvokeInput(toID, fromID, subPlan, tier string, tenure *int, multiMonthDuration int, multiMonthTenure int) []byte {
	msg := notice.Message{
		ChannelID:              toID,
		CumulativeTenureMonths: tenure,
		CustomMessage:          "",
		SubPlan:                subPlan,
		Tier:                   tier,
		Timestamp:              time.Now(),
		UserID:                 fromID,
		MultiMonthDuration:     multiMonthDuration,
		MultiMonthTenure:       multiMonthTenure,
	}

	msgBytes, _ := json.Marshal(msg)

	snsEntity := events.SNSEntity{
		Message: string(msgBytes),
	}

	snsEntityBytes, _ := json.Marshal(snsEntity)

	sqsMessage := events.SQSMessage{
		Body: string(snsEntityBytes),
	}
	records := make([]events.SQSMessage, 1, 1)
	records[0] = sqsMessage
	input := events.SQSEvent{
		Records: records,
	}

	inputBytes := new(bytes.Buffer)
	_ = json.NewEncoder(inputBytes).Encode(input)
	return inputBytes.Bytes()
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the sqs payload contains invalid json", func() {
			Convey("It should return an error", func() {
				invalidSqs := []byte("{\"Records\":}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, invalidSqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

		})

		Convey("When the sqs payload has no messages", func() {
			Convey("It should return nil", func() {
				sqs := []byte("{\"Records\":[]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the sqs message has invalid json", func() {
			Convey("It should return an error", func() {
				sqs := []byte("{\"Records\":[{\"messageId\":]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When notice event is missing the channel id", func() {
			Convey("It should return a nil result and an error", func() {
				tenure := 1
				input := getInvokeInput("", "222222", "plan", "tier", &tenure, 3, 3)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When notice event is missing the sender id", func() {
			Convey("It should return a nil result and an error", func() {
				tenure := 1
				input := getInvokeInput("111111", "", "plan", "tier", &tenure, 3, 3)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When notice event is missing the plan", func() {
			Convey("It should return a nil result and an error", func() {
				tenure := 1
				input := getInvokeInput("111111", "222222", "", "tier", &tenure, 3, 3)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When notice event is missing the tier", func() {
			Convey("It should return a nil result and an error", func() {
				tenure := 1
				input := getInvokeInput("", "222222", "plan", "", &tenure, 3, 3)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When notice event is missing the tenure", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("", "222222", "plan", "", nil, 2, 2)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: true,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	plan := "non-prime"
	tier := "tier"
	tenure := 10

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with a valid input for non-prime subs", t, func() {
		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeResubscriptionSharing).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(errors.New("dynamo failed"))

				mocks.Pubsub.AssertNotCalled(t, "PublishResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeResubscriptionSharing).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeResubscriptionSharing).Once().Return(alertStatus, nil)
				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})
			})
		})
	})

	Convey("When the handler is invoked with a valid input for prime subs", t, func() {
		plan = notice.SubPlanPrime

		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypePrimeResubscriptionSharing).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(errors.New("dynamo failed"))

				mocks.Pubsub.AssertNotCalled(t, "PublishPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypePrimeResubscriptionSharing).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 2, 2)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)
				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypePrimeResubscriptionSharing).Once().Return(alertStatus, nil)
				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishPrimeResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}

func TestInvokeWithAlertPublishingForNonPrimeSubs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: false,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	plan := "non-prime"
	tier := "tier"
	tenure := 10

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked for non prime subs", t, func() {
		Convey("It should not publish when the alerts manager says not to", func() {
			input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeResubscriptionSharing).Once().Return(alertStatus, nil)

			mocks.DynamoDB.On("InsertResubscriptionSharing", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishResubscriptionSharing", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeResubscriptionSharing).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertResubscriptionSharing", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishResubscriptionSharing", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(nil)

			Convey("It should publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithAlertPublishingForPrimeSubs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	targetUserID := "333333"
	fromUserID := "444444"
	plan := notice.SubPlanPrime
	tier := "tier2"
	tenure := 10

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked for prime subs", t, func() {
		alertStatusFalse := alerts.Status{
			CanPublish: false,
			StatusName: dynamodb.AlertStatusQueued,
		}

		Convey("It should not publish when the alerts manager says not to", func() {
			input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypePrimeResubscriptionSharing).Once().Return(alertStatusFalse, nil)

			mocks.DynamoDB.On("InsertPrimeResubscriptionSharing", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishPrimeResubscriptionSharing", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			input := getInvokeInput(targetUserID, fromUserID, plan, tier, &tenure, 3, 3)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Zuma.On("EnforceMessage", mock.Anything, targetUserID, fromUserID, mock.Anything).Once().Return("", []types.Fragment{}, nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypePrimeResubscriptionSharing).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertPrimeResubscriptionSharing", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishPrimeResubscriptionSharing", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)

			Convey("It should publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}
