// editable

package follow

import (
	"code.justin.tv/cb/sauron/types"
	"context"

	"code.justin.tv/cb/sauron/internal/clients/users"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/user_follow_user"
	log "github.com/sirupsen/logrus"
)

// processMessage handles the following steps:
// 1. validates the message
// 2. gets data for the user, based on the userId, if applicable
// 3. insert rate limit table and determine whether to dedupe, if applicable
// 4. insert the event into dynamo
// 5. publish the event to the activity feed pubsub topic
// 6. publish the even to the spotlight alerts pubsub topic, if required
func (h Handler) processMessage(ctx context.Context, header *eventbus.Header, event *user_follow_user.UserFollowUserCreate) error {
	logger := log.WithField("event", event)

	if err := validate(event); err != nil {
		h.Statsd.GoIncrement(validateErrorStat, 1)
		logger.WithError(err).Warn("follow: failed to validate message")
		return nil
	}
	follower, err := h.Users.GetUser(ctx, event.FromUserId)
	if err != nil {
		switch err {
		case users.ErrUserNotFound:
			h.Statsd.GoIncrement(userNotFoundStat, 1)
			logger.Warnf("follow: Follower not found for user id '%s'", event.FromUserId)
			return nil
		default:
			h.Statsd.GoIncrement(errorStat, 1)
			logger.WithError(err).Errorf("follow: failed to fetch Follower with user id '%s'", event.FromUserId)
			return err
		}
	}

	return h.rateLimitOrInsertAndPublish(ctx, follower, header, event)
}

// insertAndPublish handles rate limiting the event, inserting the event into dynamo, publishing the event as an activity feed item,
// and publishing the event as a spotlight alert, if necessary. This method assumes that the followMessage
// has been validated properly.
// The order of operations matters here.
// We only insert the event into the channel activity dynamo table if it is not rate limited.
// We only publish the event if we succeed in inserting to dynamo. Similarly,
// we only publish it as an alert if we succeeded to publish it as activity.
//
// When an error occurs, in the step to 1) insert into the channel activity table 2) publish follow channel activity
// or 3) publish follow alerts. We delete the entry from the rate limit table for the follow/follow pair.
// This allows us to fall back to a state without rate limiting, allowing follows to go through.
// This is better than blocking follows when there is an error.
// It is worth noting that intermittent failures that triggers DeleteRateLimitFollow
// could potentially result follows (that should have been rate limited) to go through,
// as well as inconsistencies between the channel activity table, activity pubsub and alerts.
func (h *Handler) rateLimitOrInsertAndPublish(ctx context.Context, follower types.User, header *eventbus.Header, event *user_follow_user.UserFollowUserCreate) error {
	logger := log.WithFields(log.Fields{
		"follower": follower,
		"event":    event,
	})

	timestamp := header.CreatedAt
	channelID := event.ToUserId
	followerID := follower.ID

	// Determine whether follow event should be inserted and published in activity feed and as an alert,
	// this is done by referencing the rate limit table to check whether the follow event was rate limited
	shouldInsertAndPublish, err := h.DynamoDB.ShouldInsertAndPublishFollow(ctx, channelID, followerID, timestamp)
	if err != nil {
		logger.WithError(err).Errorf("follow: unexpected rate limit error - '%s' tried to follow '%s'", followerID, channelID)
		h.Statsd.GoIncrement(statPrefix+"ratelimit.put.error", 1)
		return err
	}

	h.Statsd.GoIncrement(statPrefix+"ratelimit.put.success", 1)

	if !shouldInsertAndPublish {
		h.Statsd.GoIncrement(statPrefix+"dupe", 1)
		return nil
	}

	err = h.insertAndPublishFollow(ctx, logger, header.CreatedAt, event.ToUserId, follower)
	if err != nil {
		h.Statsd.GoOther(func() { h.DynamoDB.DeleteRateLimitFollow(context.Background(), channelID, followerID) })
		return err
	}

	return nil
}
