// editable

package follow_test

import (
	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/types"
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/eventbus/follow"
	"code.justin.tv/cb/sauron/internal/mocks"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/user_follow_user"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB     *mocks.Database
	Pubsub       *mocks.Publisher
	Users        *mocks.Users
	Statsd       *mocks.StatSender
	AlertManager *mocks.Manager
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:     &mocks.Database{},
		Pubsub:       &mocks.Publisher{},
		Users:        &mocks.Users{},
		Statsd:       &mocks.StatSender{},
		AlertManager: &mocks.Manager{},
	}
}

func mockHandler(params mockHandlerParams) *follow.Handler {
	return &follow.Handler{
		DynamoDB:     params.DynamoDB,
		Pubsub:       params.Pubsub,
		Users:        params.Users,
		Statsd:       params.Statsd,
		AlertManager: params.AlertManager,
	}
}

func getInvokeInput(toID, fromID, eventType string) (*eventbus.Header, *user_follow_user.UserFollowUserCreate) {
	event := &user_follow_user.UserFollowUserCreate{
		ToUserId:   toID,
		FromUserId: fromID,
	}

	header := &eventbus.Header{
		EventType: eventType,
		CreatedAt: time.Now(),
	}
	return header, event
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	handlerMocks := handlerMocks()
	handler := mockHandler(handlerMocks)
	ctx := context.Background()
	handlerMocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	handlerMocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the event is nil", func() {
			Convey("It should return without an error", func() {
				header, _ := getInvokeInput("111111", "222222", "UserFollowUserCreate")
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, nil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When follow event is missing the user id", func() {
			Convey("It should return a nil result and nil error", func() {
				header, event := getInvokeInput("", "222222", "UserFollowUserCreate")

				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
			})
		})

		Convey("When follow event is missing the target id", func() {
			Convey("It should return a nil result and nil error", func() {
				header, event := getInvokeInput("111111", "", "UserFollowUserCreate")

				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	alertStatus := alerts.Status{
		CanPublish: true,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"

	Convey("When the handler is invoked with a valid input", t, func() {
		handlerMocks := handlerMocks()
		handler := mockHandler(handlerMocks)
		ctx := context.Background()
		handlerMocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
		handlerMocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()
		handlerMocks.Statsd.On("GoOther", mock.Anything, mock.Anything).Maybe()

		Convey("When users service fails", func() {
			Convey("It should return a nil result and an error", func() {
				header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
				handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{}, errors.New("users service failed"))
				handlerMocks.AlertManager.AssertNotCalled(t, "GetAlertStatus", ctx, targetUserID, activity.TypeFollow)

				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldNotBeNil)

				handlerMocks.AlertManager.AssertExpectations(t)
			})
		})

		Convey("When rate limitted follow actvity; insert into rate limit table errors", func() {
			Convey("It should return nil result and an error, and not publish", func() {
				header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
				handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatus, nil)

				handlerMocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(false, errors.New("rate limitted"))
				handlerMocks.DynamoDB.On("DeleteRateLimitFollow", ctx, targetUserID, fromUserID).Maybe()
				handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				handlerMocks.DynamoDB.AssertNotCalled(t, "DeleteRateLimitFollow", mock.Anything, mock.Anything, mock.Anything)

				handlerMocks.Pubsub.AssertNotCalled(t, "PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything)

				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
				handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatus, nil)

				handlerMocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(true, nil)
				handlerMocks.DynamoDB.On("DeleteRateLimitFollow", ctx, targetUserID, fromUserID).Maybe()
				handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(errors.New("dynamo failed"))

				handlerMocks.Pubsub.AssertNotCalled(t, "PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything)

				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
				handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatus, nil)

				handlerMocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(true, nil)
				handlerMocks.DynamoDB.On("DeleteRateLimitFollow", ctx, targetUserID, fromUserID).Maybe()
				handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				handlerMocks.Pubsub.On("PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {

					handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					err := handler.Handle(ctx, header, event)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
				handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatus, nil)

				handlerMocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(true, nil)
				handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				handlerMocks.Pubsub.On("PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
				handlerMocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					err := handler.Handle(ctx, header, event)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}

func TestInvokeWithAlertPublishing(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	handlerMocks := handlerMocks()
	handler := mockHandler(handlerMocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: false,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	handlerMocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	handlerMocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked", t, func() {
		Convey("It should not publish when the alerts manager says not to", func() {
			header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
			handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)

			handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatus, nil)
			handlerMocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)

			handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(nil)
			handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(true, nil)
			handlerMocks.Pubsub.On("PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
				handlerMocks.AlertManager.AssertExpectations(t)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			header, event := getInvokeInput(targetUserID, fromUserID, "UserFollowUserCreate")
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			handlerMocks.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			handlerMocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeFollow).Once().Return(alertStatusSuccess, nil)
			handlerMocks.DynamoDB.On("InsertFollow", ctx, mock.Anything, mock.Anything).Once().Return(nil)
			handlerMocks.Pubsub.On("PublishFollow", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			handlerMocks.DynamoDB.On("ShouldInsertAndPublishFollow", ctx, targetUserID, fromUserID, mock.Anything).Once().Return(true, nil)
			handlerMocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)
			Convey("It should not publish an alert", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
			})
		})
	})
}
