package raiding

import (
	"context"

	"code.justin.tv/cb/sauron/internal/clients/users"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/raid"
	log "github.com/sirupsen/logrus"
)

// processMessage handles the following steps:
// 1. validates the message
// 2. gets data for the user, based on the userId, if applicable
// 3. insert rate limit table and determine whether to dedupe, if applicable
// 4. insert the event into dynamo
// 5. publish the event to the activity feed pubsub topic
// 6. publish the even to the spotlight alerts pubsub topic, if required
func (h Handler) processMessage(ctx context.Context, header *eventbus.Header, event *raid.RaidUpdate) error {
	logger := log.WithField("event", event)

	if err := validate(event); err != nil {
		h.Statsd.GoIncrement(validateErrorStat, 1)
		logger.WithError(err).Warn("raiding: failed to validate message")
		return nil
	}
	user, err := h.Users.GetUser(ctx, event.FromBroadcasterUserId)
	if err != nil {
		switch err {
		case users.ErrUserNotFound:
			h.Statsd.GoIncrement(userNotFoundStat, 1)
			logger.Warnf("raiding: Raider not found for user id '%s'", event.FromBroadcasterUserId)
			return nil
		default:
			h.Statsd.GoIncrement(errorStat, 1)
			logger.WithError(err).Errorf("raiding: failed to fetch Raider with user id '%s'", event.FromBroadcasterUserId)
			return err
		}
	}

	logger = log.WithFields(log.Fields{
		"raiding_event": event,
		"raider":        user,
	})

	return h.insertAndPublishRaiding(ctx, logger, header.CreatedAt, event.ToBroadcasterUserId, user, event.Viewers, int32(event.Status))
}
