package raiding_test

import (
	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/types"
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/internal/eventbus/raiding"
	"code.justin.tv/cb/sauron/internal/mocks"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/raid"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB     *mocks.Database
	Pubsub       *mocks.Publisher
	Users        *mocks.Users
	Statsd       *mocks.StatSender
	AlertManager *mocks.Manager
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:     &mocks.Database{},
		Pubsub:       &mocks.Publisher{},
		Users:        &mocks.Users{},
		Statsd:       &mocks.StatSender{},
		AlertManager: &mocks.Manager{},
	}
}

func mockHandler(params mockHandlerParams) *raiding.Handler {
	return &raiding.Handler{
		DynamoDB:     params.DynamoDB,
		Pubsub:       params.Pubsub,
		Users:        params.Users,
		Statsd:       params.Statsd,
		AlertManager: params.AlertManager,
	}
}

func getInvokeInput(toID string, eventType string, RaiderID string, Status raid.Status, RaidingViewerCount int32) (*eventbus.Header, *raid.RaidUpdate) {
	event := &raid.RaidUpdate{
		ToBroadcasterUserId:   toID,
		FromBroadcasterUserId: RaiderID,
		Status:                Status,
		Viewers:               RaidingViewerCount,
	}

	header := &eventbus.Header{
		EventType: eventType,
		CreatedAt: time.Now(),
	}

	return header, event
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	m := handlerMocks()
	handler := mockHandler(m)
	ctx := context.Background()
	m.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	m.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the event is nil", func() {
			Convey("It should return nil error", func() {
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, nil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the event has no user ID", func() {
			Convey("It should return nil error", func() {
				_, event := getInvokeInput("", "", "", 1, 1)
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, event)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	alertStatus := alerts.Status{
		CanPublish: true,
		StatusName: dynamodb.AlertStatusQueued,
	}
	m := handlerMocks()
	handler := mockHandler(m)
	ctx := context.Background()
	m.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	m.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	targetUserID := "111111"
	fromUserID := "222222"

	Convey("When the handler is invoked with a valid input", t, func() {
		Convey("When users service fails", func() {
			Convey("It should return a nil result and an error", func() {
				header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
				m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{}, errors.New("users service failed"))
				m.AlertManager.AssertNotCalled(t, "GetAlertStatus", ctx, targetUserID, activity.TypeRaiding)

				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldNotBeNil)

				m.AlertManager.AssertExpectations(t)
			})
		})

		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
				m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				m.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeRaiding).Once().Return(alertStatus, nil)

				m.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				m.DynamoDB.On("InsertRaiding", ctx, mock.Anything, mock.Anything).Once().Return(errors.New("dynamo failed"))

				m.Pubsub.AssertNotCalled(t, "PublishRaiding", ctx, mock.Anything, mock.Anything, mock.Anything)

				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
				m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				m.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeRaiding).Once().Return(alertStatus, nil)

				m.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				m.DynamoDB.On("InsertRaiding", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				m.Pubsub.On("PublishRaiding", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {

					m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					err := handler.Handle(ctx, header, event)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
				m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				m.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeRaiding).Once().Return(alertStatus, nil)

				m.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				m.DynamoDB.On("InsertRaiding", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				m.Pubsub.On("PublishRaiding", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
				m.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					err := handler.Handle(ctx, header, event)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}
func TestInvokeWithAlertPublishing(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	m := handlerMocks()
	handler := mockHandler(m)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: false,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	m.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	m.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked", t, func() {
		Convey("It should not publish when the alerts manager says not to", func() {
			header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
			m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)

			m.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeRaiding).Once().Return(alertStatus, nil)
			m.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)

			m.DynamoDB.On("InsertRaiding", ctx, mock.Anything, mock.Anything).Once().Return(nil)
			m.Pubsub.On("PublishRaiding", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
				m.AlertManager.AssertExpectations(t)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			header, event := getInvokeInput(targetUserID, "RaidUpdate", fromUserID, raid.Status_STATUS_STARTED, 120)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			m.Users.On("GetUser", ctx, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			m.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeRaiding).Once().Return(alertStatusSuccess, nil)
			m.DynamoDB.On("InsertRaiding", ctx, mock.Anything, mock.Anything).Once().Return(nil)
			m.Pubsub.On("PublishRaiding", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			m.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)
			Convey("It should not publish an alert", func() {
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, header, event)
				So(err, ShouldBeNil)
			})
		})
	})
}
