package userdestroy

import (
	"context"
	"time"

	"code.justin.tv/cb/sauron/internal/clients/pdms"

	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/user"

	log "github.com/sirupsen/logrus"
)

// Handler implements github.com/aws/aws-lambda-go/lambda.Handler.
type Handler struct {
	DynamoDB   dynamodb.Database
	Statsd     stats.StatSender
	PdmsClient pdms.ClientAPI
}

func (h Handler) Handle(ctx context.Context, header *eventbus.Header, event *user.Destroy) error {
	start := time.Now()
	defer func() {
		since := time.Since(start)
		h.Statsd.GoExecutionTime("userdestroy.duration", since)
		cancelCtx, cancel := context.WithTimeout(ctx, 5*time.Second)
		defer cancel()
		if err := h.Statsd.Shutdown(cancelCtx); err != nil {
			log.WithError(err).Warn("timeout before statsd finished")
		}
	}()

	logger := log.WithField("event", event).WithField("header", header)

	if event == nil {
		logger.Warn("userdestroy: nil eventbus event")
		return nil
	}

	userID := event.GetUserId()
	if userID == "" {
		logger.Warn("userdestroy: userID cannot be empty")
		return nil
	}

	err := h.DynamoDB.DeleteActivities(ctx, userID)
	if err != nil {
		logger.WithError(err).Error("userdestroy: failed to delete activities from dynamodb")
		h.Statsd.GoIncrement("userdestroy.error", 1)
		return err
	}

	err = h.DynamoDB.DeleteAlertPreferences(ctx, userID)
	if err != nil {
		logger.WithError(err).Error("userdestroy: failed to delete alert preferences from dynamodb")
		h.Statsd.GoIncrement("userdestroy.error", 1)
		return err
	}

	err = h.PdmsClient.ReportDeletion(ctx, userID)
	if err != nil {
		logger.WithError(err).Error("userdestroy: failed to report user deletion to PDMS")
		h.Statsd.GoIncrement("userdestroy.error", 1)
		return err
	}

	logger.Info("userdestroy: user destroyed for compliance")
	h.Statsd.GoIncrement("userdestroy.success", 1)
	return nil
}
