package userdestroy_test

import (
	userdestroy2 "code.justin.tv/cb/sauron/internal/eventbus/userdestroy"
	"context"
	"errors"
	"testing"

	"code.justin.tv/cb/sauron/internal/mocks"
	"code.justin.tv/eventbus/schema/pkg/user"

	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB   *mocks.Database
	Statsd     *mocks.StatSender
	PdmsClient *mocks.PdmsClient
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:   &mocks.Database{},
		Statsd:     &mocks.StatSender{},
		PdmsClient: &mocks.PdmsClient{},
	}
}

func mockHandler(params mockHandlerParams) *userdestroy2.Handler {
	return &userdestroy2.Handler{
		DynamoDB:   params.DynamoDB,
		Statsd:     params.Statsd,
		PdmsClient: params.PdmsClient,
	}
}

func getInvokeInput(userID string) *user.Destroy {
	return &user.Destroy{
		UserId: userID,
	}
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	handlerMocks := handlerMocks()
	handler := mockHandler(handlerMocks)
	ctx := context.Background()
	handlerMocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	handlerMocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the event is nil", func() {
			Convey("It should return nil error", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, nil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the event has no user ID", func() {
			Convey("It should return nil error", func() {
				event := getInvokeInput("")
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, event)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	handlerMocks := handlerMocks()
	handler := mockHandler(handlerMocks)
	ctx := context.Background()
	userID := "111111"
	handlerMocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	handlerMocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with a valid input", t, func() {
		Convey("When deleting activities fails", func() {
			event := getInvokeInput(userID)

			handlerMocks.DynamoDB.On("DeleteActivities", ctx, userID).Once().Return(errors.New("dynamo failed"))

			Convey("It should return an error", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, event)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When deleting alert preferences fails", func() {
			event := getInvokeInput(userID)

			handlerMocks.DynamoDB.On("DeleteActivities", ctx, userID).Once().Return(nil)
			handlerMocks.DynamoDB.On("DeleteAlertPreferences", ctx, userID).Once().Return(errors.New("dynamo failed"))

			Convey("It should return an error", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, event)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When deleting succeeds", func() {
			event := getInvokeInput(userID)

			handlerMocks.DynamoDB.On("DeleteActivities", ctx, userID).Once().Return(nil)
			handlerMocks.DynamoDB.On("DeleteAlertPreferences", ctx, userID).Once().Return(nil)
			handlerMocks.PdmsClient.On("ReportDeletion", ctx, userID).Once().Return(nil)

			Convey("It should return a nil error", func() {
				handlerMocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				err := handler.Handle(ctx, nil, event)
				So(err, ShouldBeNil)
			})
		})
	})
}
