package sauronserver

import (
	"context"
	"errors"

	"code.justin.tv/cb/sauron/activity/api"
	"code.justin.tv/cb/sauron/internal/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	"code.justin.tv/cb/sauron/internal/convert"

	serrors "code.justin.tv/cb/sauron/internal/errors"
	pb "code.justin.tv/cb/sauron/rpc/sauron"
	log "github.com/sirupsen/logrus"
)

// Server implements the Sauron server
type Server struct {
	DynamoDB     dynamodb.Database
	Statsd       stats.StatSender
	AlertManager alerts.Manager
}

func (s *Server) GetActivities(ctx context.Context, req *pb.GetActivitiesReq) (*pb.GetActivitiesResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Warning("alerts: get activities request contained no channel id")
		return nil, serrors.ErrMissingChannelID
	}

	limit := req.GetLimit()
	if limit < 0 {
		log.Warning("alerts: get activities request contained invalid limit")
		return nil, serrors.ErrInvalidLimit
	}

	handlerInput := api.GetInput{
		ChannelID: channelID,
		Limit:     int(limit),
	}

	cursor := req.GetCursor()
	if cursor != "" {
		handlerInput.Cursor = &cursor
	}

	handler := &activity.Handler{
		DynamoDB: s.DynamoDB,
		Statsd:   s.Statsd,
	}

	result, err := handler.Handle(ctx, handlerInput)
	if err != nil {
		log.WithError(err).Error("activities: failed to retrieve activities")
		return nil, err
	}

	return buildActivitiesResp(result), nil
}

// GetAlertPrefs returns the alert preferences for a user
func (s *Server) GetAlertPrefs(ctx context.Context, req *pb.GetAlertPrefsReq) (*pb.GetAlertPrefsResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Warning("alerts: get preferences request contained no channel id")
		return nil, serrors.ErrMissingChannelID
	}

	alertPrefs, err := s.AlertManager.GetAlertPrefs(ctx, channelID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"channel_id": channelID,
		}).Error("alerts: failed to get preferences")

		return nil, err
	}

	return &pb.GetAlertPrefsResp{
		AlertPrefs: convert.DynamoAlertPrefsToPB(alertPrefs),
	}, nil
}

// SetAlertPrefs sets a given alert preference for a user
func (s *Server) SetAlertPrefs(ctx context.Context, req *pb.SetAlertPrefsReq) (*pb.SetAlertPrefsResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Warning("alerts: set preferences request contained no channel id")
		return nil, serrors.ErrMissingChannelID
	}

	key := req.GetKey()
	if len(key) == 0 {
		log.WithField("channel_id", req.ChannelID).Error("alerts: set preference request contained no preferences")
		return nil, serrors.ErrInvalidPreferenceKey
	}
	val := req.GetVal()

	alertPrefs, err := s.AlertManager.SetAlertPrefs(ctx, channelID, key, val)
	if err != nil {
		log.WithField("channel_id", channelID).WithError(err).Error("alerts: failed to write alert preference")
		return nil, err
	}

	return &pb.SetAlertPrefsResp{
		AlertPrefs: convert.DynamoAlertPrefsToPB(alertPrefs),
	}, nil
}

func (s *Server) GetAlertQueue(ctx context.Context, req *pb.GetAlertQueueReq) (*pb.GetAlertQueueResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		return nil, serrors.ErrMissingChannelID
	}

	cursor := req.GetCursor()
	before, err := convert.AlertCursorToTime(cursor)
	if err != nil {
		return nil, serrors.ErrInvalidCursor
	}

	limit := int(req.GetLimit())
	if limit < 0 {
		log.Warning("alerts: get alert queue request contained invalid limit")
		return nil, serrors.ErrInvalidLimit
	}
	if limit < 1 || limit > dynamodb.MaxAlertLimit {
		limit = dynamodb.MaxAlertLimit
	}

	alerts, err := s.DynamoDB.GetAlertQueue(ctx, channelID, before, limit)
	if err != nil {
		log.WithField("channel_id", channelID).WithError(err).Error("alerts: failed to get alert queue")
		return nil, err
	}

	return buildAlertsResp(alerts), nil
}

func (s *Server) SetAlertStatus(ctx context.Context, req *pb.SetAlertStatusReq) (*pb.SetAlertStatusResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		return nil, serrors.ErrMissingChannelID
	}

	activityID := req.GetActivityID()
	if len(activityID) == 0 {
		return nil, serrors.ErrMissingActivityID
	}

	alertStatus := req.GetAlertStatus()
	if len(alertStatus) == 0 {
		return nil, serrors.ErrInvalidAlertStatus
	}

	activity, err := s.AlertManager.UpdateAndPublishAlert(ctx, channelID, activityID, alertStatus)
	if err != nil {
		log.WithFields(log.Fields{
			"channel_id":   channelID,
			"activity_id":  activityID,
			"alert_status": alertStatus,
		}).WithError(err).Error("failed to set alert status")
		return nil, err
	}

	if activity == nil {
		return nil, errors.New("alerts: failed to get updated alert")
	}

	return &pb.SetAlertStatusResp{
		Activity: convert.DynamoActivityToPB(*activity),
	}, nil
}

func buildAlertsResp(alerts []dynamodb.Activity) *pb.GetAlertQueueResp {
	newActivities := make([]*pb.Activity, len(alerts))

	for idx, alert := range alerts {
		activity := alert
		newActivities[idx] = convert.DynamoActivityToPB(activity)
	}

	return &pb.GetAlertQueueResp{
		Alerts: newActivities,
	}
}

func buildActivitiesResp(activity *api.GetOutput) *pb.GetActivitiesResp {
	newActivities := make([]*pb.Activity, len(activity.Activities))

	for idx, activity := range activity.Activities {
		newActivities[idx] = convert.ActivityToPB(activity)
	}

	return &pb.GetActivitiesResp{
		Activities: newActivities,
	}
}
