package sauronserver_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/mocks"
	"code.justin.tv/cb/sauron/internal/sauronserver"

	"github.com/stretchr/testify/mock"

	serrors "code.justin.tv/cb/sauron/internal/errors"
	pb "code.justin.tv/cb/sauron/rpc/sauron"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

type mockServerParams struct {
	DynamoDB     *mocks.Database
	Statsd       *mocks.StatSender
	AlertManager *mocks.Manager
}

func serverMocks() mockServerParams {
	return mockServerParams{
		DynamoDB:     &mocks.Database{},
		Statsd:       &mocks.StatSender{},
		AlertManager: &mocks.Manager{},
	}
}

func mockServer(params mockServerParams) *sauronserver.Server {
	return &sauronserver.Server{
		DynamoDB:     params.DynamoDB,
		Statsd:       params.Statsd,
		AlertManager: params.AlertManager,
	}
}

func TestGetAlertPrefs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When given a nil request", t, func() {
		Convey("It should return an error", func() {
			mocks.AlertManager.AssertNotCalled(t, "GetAlertPrefs", ctx, mock.Anything)

			result, err := server.GetAlertPrefs(ctx, nil)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When given an empty channel id", t, func() {
		Convey("It should return an error", func() {
			channelID := ""
			req := &pb.GetAlertPrefsReq{
				ChannelID: channelID,
			}
			mocks.AlertManager.AssertNotCalled(t, "GetAlertPrefs", ctx, channelID)

			result, err := server.GetAlertPrefs(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When given a valid request and alert manager fails", t, func() {
		Convey("It should return an error", func() {
			channelID := "111111"
			req := &pb.GetAlertPrefsReq{
				ChannelID: channelID,
			}
			mocks.AlertManager.On("GetAlertPrefs", ctx, channelID).Once().Return(nil, errors.New("alert manager failed"))

			result, err := server.GetAlertPrefs(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When given a valid request and alert manager succeeds", t, func() {
		Convey("It should return a valid response", func() {
			channelID := "222222"
			req := &pb.GetAlertPrefsReq{
				ChannelID: channelID,
			}
			alertsResp := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: true,
				HideFollows:    true,
				HideRaids:      false,
			}
			mocks.AlertManager.On("GetAlertPrefs", ctx, channelID).Once().Return(alertsResp, nil)

			result, err := server.GetAlertPrefs(ctx, req)
			So(result, ShouldNotBeNil)
			So(err, ShouldBeNil)
			So(result.AlertPrefs, ShouldNotBeNil)
			So(result.AlertPrefs.ChannelID, ShouldEqual, channelID)
			So(result.AlertPrefs.DNDModeEnabled, ShouldEqual, true)
			So(result.AlertPrefs.HideFollows, ShouldEqual, true)
			So(result.AlertPrefs.HideRaids, ShouldEqual, false)
			So(result.AlertPrefs.HideHosts, ShouldEqual, false)

			mocks.AlertManager.AssertExpectations(t)
		})
	})
}

func TestSetAlertPrefs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When given a nil request", t, func() {
		Convey("It should return an error", func() {
			mocks.AlertManager.AssertNotCalled(t, "SetAlertPrefs", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.SetAlertPrefs(ctx, nil)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When given an empty channel id", t, func() {
		Convey("It should return an error", func() {
			channelID := ""
			prefKey := "prefkey"
			prefVal := true
			req := &pb.SetAlertPrefsReq{
				ChannelID: channelID,
				Key:       prefKey,
				Val:       prefVal,
			}
			mocks.AlertManager.AssertNotCalled(t, "SetAlertPrefs", ctx, channelID, prefKey, prefVal)

			result, err := server.SetAlertPrefs(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When given an empty preference key", t, func() {
		Convey("It should return an error", func() {
			channelID := "222222"
			prefKey := ""
			prefVal := true
			req := &pb.SetAlertPrefsReq{
				ChannelID: channelID,
				Key:       prefKey,
				Val:       prefVal,
			}
			mocks.AlertManager.AssertNotCalled(t, "SetAlertPrefs", ctx, channelID, prefKey, prefVal)

			result, err := server.SetAlertPrefs(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When the alert manager fails", t, func() {
		Convey("It should return an error", func() {
			channelID := "555555"
			prefKey := "prefKey"
			prefVal := true
			req := &pb.SetAlertPrefsReq{
				ChannelID: channelID,
				Key:       prefKey,
				Val:       prefVal,
			}
			mocks.AlertManager.On("SetAlertPrefs", ctx, channelID, prefKey, prefVal).Once().Return(nil, errors.New("alert manager failed"))

			result, err := server.SetAlertPrefs(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

	Convey("When the alert manager succeeds", t, func() {
		Convey("It should return a valid respons", func() {
			channelID := "444444"
			prefKey := "prefKey"
			prefVal := true
			req := &pb.SetAlertPrefsReq{
				ChannelID: channelID,
				Key:       prefKey,
				Val:       prefVal,
			}
			alertsResp := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: true,
				HideFollows:    true,
				HideRaids:      false,
			}
			mocks.AlertManager.On("SetAlertPrefs", ctx, channelID, prefKey, prefVal).Once().Return(alertsResp, nil)

			result, err := server.SetAlertPrefs(ctx, req)
			So(result, ShouldNotBeNil)
			So(err, ShouldBeNil)

			mocks.AlertManager.AssertExpectations(t)
		})
	})

}

func TestGetAlertQueue(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When given a nil request", t, func() {
		Convey("It should return an error", func() {
			mocks.DynamoDB.AssertNotCalled(t, "GetAlertQueue", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.GetAlertQueue(ctx, nil)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When given an empty channel id", t, func() {
		Convey("It should return an error", func() {
			req := &pb.GetAlertQueueReq{
				ChannelID: "",
			}
			mocks.DynamoDB.AssertNotCalled(t, "GetAlertQueue", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.GetAlertQueue(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrMissingChannelID)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When given an invalid cursor", t, func() {
		Convey("It should return an error", func() {
			req := &pb.GetAlertQueueReq{
				ChannelID: "111111",
				Cursor:    "not-a-timestamp",
			}
			mocks.DynamoDB.AssertNotCalled(t, "GetAlertQueue", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.GetAlertQueue(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrInvalidCursor)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When given an invalid limit", t, func() {
		Convey("It should return an error", func() {
			req := &pb.GetAlertQueueReq{
				ChannelID: "111111",
				Cursor:    "",
				Limit:     -10,
			}
			mocks.DynamoDB.AssertNotCalled(t, "GetAlertQueue", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.GetAlertQueue(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrInvalidLimit)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When dynamodb fails", t, func() {
		Convey("It should return an error", func() {
			channelID := "123456"
			req := &pb.GetAlertQueueReq{
				ChannelID: channelID,
			}
			mocks.DynamoDB.On("GetAlertQueue", ctx, channelID, mock.Anything, mock.Anything).Once().Return(nil, errors.New("dynamo error"))

			result, err := server.GetAlertQueue(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("On a valid request", t, func() {
		Convey("It returns a valid response containing the alert queue", func() {
			channelID := "111111"
			status := dynamodb.AlertStatusQueued
			dynamoResp := []dynamodb.Activity{
				{
					ChannelID:   channelID,
					Timestamp:   time.Now(),
					Type:        activity.TypeFollow,
					AlertStatus: &status,
				},
				{
					ChannelID:   channelID,
					Timestamp:   time.Now(),
					Type:        activity.TypeFollow,
					AlertStatus: &status,
				},
			}
			mocks.DynamoDB.On("GetAlertQueue", ctx, channelID, mock.Anything, mock.Anything).Once().Return(dynamoResp, nil)

			req := &pb.GetAlertQueueReq{
				ChannelID: channelID,
			}
			result, err := server.GetAlertQueue(ctx, req)
			So(result, ShouldNotBeNil)
			So(err, ShouldBeNil)
			So(result.Alerts, ShouldNotBeNil)
			So(len(result.Alerts), ShouldEqual, 2)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})
}

func TestSetAlertStatus(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When given an empty channel id", t, func() {
		Convey("It should return an error", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID: "",
			}
			mocks.AlertManager.AssertNotCalled(t, "UpdateAndPublishAlert", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrMissingChannelID)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When given an invalid activity id", t, func() {
		Convey("It should return an error", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID:  "333333",
				ActivityID: "",
			}
			mocks.AlertManager.AssertNotCalled(t, "UpdateAndPublishAlert", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrMissingActivityID)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When given an empty alert status", t, func() {
		Convey("It should return an error", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID:   "111111",
				ActivityID:  "123456",
				AlertStatus: "",
			}
			mocks.AlertManager.AssertNotCalled(t, "UpdateAndPublishAlert", ctx, mock.Anything, mock.Anything, mock.Anything)

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, serrors.ErrInvalidAlertStatus)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When alert manager fails", t, func() {
		Convey("It should return an error", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID:   "111111",
				ActivityID:  "123456",
				AlertStatus: "skipped",
			}
			mocks.AlertManager.On("UpdateAndPublishAlert", ctx, req.ChannelID, req.ActivityID, req.AlertStatus).Once().Return(nil, errors.New("alerts error"))

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When alert manager succeeds but returns a nil activity struct", t, func() {
		Convey("It should return an error", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID:   "555555",
				ActivityID:  "123456",
				AlertStatus: "skipped",
			}
			mocks.AlertManager.On("UpdateAndPublishAlert", ctx, req.ChannelID, req.ActivityID, req.AlertStatus).Once().Return(nil, nil)

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldBeNil)
			So(err, ShouldNotBeNil)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})

	Convey("When alert manager succeeds", t, func() {
		Convey("It should return a valid response with activity", func() {
			req := &pb.SetAlertStatusReq{
				ChannelID:   "555555",
				ActivityID:  "123456",
				AlertStatus: "skipped",
			}
			followerID := "000000"
			alertsResp := &dynamodb.Activity{
				ID:         "abc",
				Type:       activity.TypeFollow,
				Timestamp:  time.Now(),
				FollowerID: &followerID,
			}
			mocks.AlertManager.On("UpdateAndPublishAlert", ctx, req.ChannelID, req.ActivityID, req.AlertStatus).Once().Return(alertsResp, nil)

			result, err := server.SetAlertStatus(ctx, req)
			So(result, ShouldNotBeNil)
			So(err, ShouldBeNil)

			mocks.DynamoDB.AssertExpectations(t)
		})
	})
}
