# Cannot Terraform a Lambda function without either `filename` or `s3_*`,
# so we have to do a one-time building and zipping of the Go executable in order to apply this Terraform provisioning:
resource "null_resource" "dummy_build" {
  provisioner "local-exec" {
    working_dir = "../../"
    command     = "GOOS=linux go build -o .build/${local.api_lambda_handler_name} cmd/dummy/main.go"
  }

  # Locally execute `zip` instead of using `data.archive_file`,
  # so that Terraform does not "read (data resources)" from `data.archive_file` every run.
  provisioner "local-exec" {
    working_dir = "../../.build/"
    command     = "zip ${local.api_lambda_handler_name}.zip ${local.api_lambda_handler_name}"
  }
}

resource "aws_lambda_function" "api_handler" {
  depends_on = [null_resource.dummy_build]

  # Note that your function-name must match the name of your Lambda handler name. https://docs.aws.amazon.com/lambda/latest/dg/lambda-go-how-to-create-deployment-package.html
  function_name = local.api_lambda_handler_name
  handler       = local.api_lambda_handler_name

  description                    = var.lambda_description
  filename                       = "../../.build/${local.api_lambda_handler_name}.zip"
  role                           = var.lambda_iam_role_arn
  runtime                        = "go1.x"
  reserved_concurrent_executions = var.lambda_reserved_concurrent_executions

  vpc_config {
    subnet_ids         = split(",", data.terraform_remote_state.account.outputs.private_subnets)
    security_group_ids = [data.terraform_remote_state.account.outputs.twitch_subnets_sg]
  }

  environment {
    variables = {
      DYNAMODB_TABLE_NAME       = var.dynamodb_table_name
      ENVIRONMENT               = var.environment
      STATSD_HOST               = var.statsd_host
      ROLLBAR_TOKEN_SECRET_KEY  = var.rollbar_token_secret_key
      ROLLBAR_TOKEN_SECRET_NAME = var.rollbar_token_secret_name
    }
  }

  tags = {
    environment = var.environment
    name        = var.service_name
    owner       = var.owner
    project     = var.project
    team        = var.team_name
  }
}

resource "aws_lambda_permission" "edge_graphql_invocation" {
  statement_id  = "allow_edge_graphql_aws_account_to_invoke"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.api_handler.function_name
  principal     = var.edge_graphql_aws_account_arn
}
