# Cannot Terraform a Lambda function without either `filename` or `s3_*`,
# so we have to do a one-time building and zipping of the Go executable in order to apply this Terraform provisioning:
resource "null_resource" "dummy_build" {
  provisioner "local-exec" {
    working_dir = "../../"
    command     = "GOOS=linux go build -o .build/${local.sqs_lambda_handler_name} cmd/dummy/main.go"
  }

  # Locally execute `zip` instead of using `data.archive_file`,
  # so that Terraform does not "read (data resources)" from `data.archive_file` every run.
  provisioner "local-exec" {
    working_dir = "../../.build/"
    command     = "zip ${local.sqs_lambda_handler_name}.zip ${local.sqs_lambda_handler_name}"
  }
}

resource "aws_sqs_queue" "dlq" {
  name = "${local.sqs_lambda_handler_name}-dlq"
  # Use one day for DLQ for now, so messages don't dwell and build up for too long of a period.
  message_retention_seconds = 86400
}

resource "aws_lambda_function" "sqs_handler" {
  depends_on = [null_resource.dummy_build]

  # Note that your function-name must match the name of your Lambda handler name. https://docs.aws.amazon.com/lambda/latest/dg/lambda-go-how-to-create-deployment-package.html
  function_name = local.sqs_lambda_handler_name
  handler       = local.sqs_lambda_handler_name

  description                    = "Handles SNS messages from ${var.sns_topic_arn} via SQS"
  filename                       = "../../.build/${local.sqs_lambda_handler_name}.zip"
  role                           = var.lambda_iam_role_arn
  runtime                        = "go1.x"
  reserved_concurrent_executions = var.lambda_reserved_concurrent_executions
  timeout                        = var.timeout

  vpc_config {
    subnet_ids         = split(",", data.terraform_remote_state.account.outputs.private_subnets)
    security_group_ids = [data.terraform_remote_state.account.outputs.twitch_subnets_sg]
  }

  environment {
    variables = {
      ALERT_PREFERENCES_TABLE_NAME = var.alert_preferences_table_name
      DYNAMODB_TABLE_NAME          = var.dynamodb_table_name
      ENVIRONMENT                  = var.environment
      LIVELINE_HOST                = var.liveline_host
      PUBSUB_HOST                  = var.pubsub_host
      RATE_LIMIT_TABLE_NAME        = var.rate_limit_table_name
      ROLLBAR_TOKEN_SECRET_KEY     = var.rollbar_token_secret_key
      ROLLBAR_TOKEN_SECRET_NAME    = var.rollbar_token_secret_name
      STATSD_HOST                  = var.statsd_host
      SUBSCRIPTIONS_HOST           = var.subscriptions_host
      COPO_HOST                    = var.copo_host
      USERS_SERVICE_HOST           = var.users_service_host
      ZUMA_HOST                    = var.zuma_host
    }
  }

  tags = {
    environment = var.environment
    name        = var.service_name
    owner       = var.owner
    project     = var.project
    team        = var.team_name
  }

  dead_letter_config {
    target_arn = aws_sqs_queue.dlq.arn
  }
}

resource "aws_lambda_event_source_mapping" "sqs" {
  batch_size       = 1
  event_source_arn = aws_sqs_queue.sns_subscriber.arn
  function_name    = aws_lambda_function.sqs_handler.function_name
}

resource "aws_lambda_permission" "sqs" {
  statement_id  = "allow_sqs_to_invoke"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.sqs_handler.function_name
  principal     = "sqs.amazonaws.com"
  source_arn    = aws_sqs_queue.sns_subscriber.arn
}
