resource "aws_lambda_function" "populate_NLB_TG_with_ALB" {
  function_name = "${var.team_code}-${var.name}-${var.environment}-private-link"
  s3_bucket     = "exampleloadbalancer-us-west-2"
  s3_key        = "blog-posts/static-ip-for-application-load-balancer/populate_NLB_TG_with_ALB.zip"
  description   = "Register Application Load Balancer to Network Load Balancer"
  handler       = "populate_NLB_TG_with_ALB.lambda_handler"
  role          = aws_iam_role.populate_NLB_TG_with_ALB.arn
  runtime       = "python2.7"
  timeout       = "300"

  environment {
    variables = {
      ALB_DNS_NAME                      = var.beanstalk_loadbalancer_name
      ALB_LISTENER                      = "80"
      CW_METRIC_FLAG_IP_COUNT           = "True"
      INVOCATIONS_BEFORE_DEREGISTRATION = "3"
      MAX_LOOKUP_PER_INVOCATION         = "50"
      NLB_TG_ARN                        = aws_lb_target_group.pl_target_group.arn
      S3_BUCKET                         = aws_s3_bucket.elb_ips_bucket.id
    }
  }
}

resource "aws_iam_role" "populate_NLB_TG_with_ALB" {
  name = "${var.team_code}-${var.name}-${var.environment}-private-link"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
}
EOF
}

data "aws_iam_policy_document" "populate_NLB_TG_with_ALB" {
  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = ["arn:aws:logs:*:*:*"]
  }

  statement {
    actions = [
      "s3:Get*",
      "s3:ListBucket",
    ]

    resources = [aws_s3_bucket.elb_ips_bucket.arn]
  }

  statement {
    actions = [
      "s3:GetObject",
      "s3:PutObject",
    ]

    resources = ["${aws_s3_bucket.elb_ips_bucket.arn}/*"]
  }

  statement {
    actions   = ["elasticloadbalancing:Describe*"]
    resources = ["*"]
  }

  statement {
    actions = [
      "elasticloadbalancing:RegisterTargets",
      "elasticloadbalancing:DeregisterTargets",
    ]

    resources = [aws_lb_target_group.pl_target_group.arn]
  }

  statement {
    actions   = ["cloudwatch:putMetricData"]
    resources = ["*"]
  }
}

resource "aws_iam_role_policy" "populate_NLB_TG_with_ALB" {
  name   = "Lambda-ALBasTarget"
  role   = aws_iam_role.populate_NLB_TG_with_ALB.id
  policy = data.aws_iam_policy_document.populate_NLB_TG_with_ALB.json
}

resource "aws_lambda_permission" "populate_NLB_TG_with_ALB" {
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.populate_NLB_TG_with_ALB.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.pl_event_rule.arn
}

resource "aws_cloudwatch_event_rule" "pl_event_rule" {
  name                = "${aws_lambda_function.populate_NLB_TG_with_ALB.function_name}-scheduler"
  schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "pl_event_target" {
  rule = aws_cloudwatch_event_rule.pl_event_rule.name
  arn  = aws_lambda_function.populate_NLB_TG_with_ALB.arn
}

resource "aws_s3_bucket" "elb_ips_bucket" {
  bucket = "${var.team_code}-${var.name}-${var.environment}-elb-ips"
  acl    = "private"
}

resource "aws_lb" "pl_nlb" {
  name                             = "${var.team_code}-${var.name}-${var.environment}-nlb"
  internal                         = true
  load_balancer_type               = "network"
  enable_cross_zone_load_balancing = true

  subnets = split(
    ",",
    data.terraform_remote_state.account.outputs.private_subnets,
  )
}

resource "aws_lb_target_group" "pl_target_group" {
  name                 = aws_lb.pl_nlb.name
  port                 = 80
  protocol             = "TCP"
  vpc_id               = data.terraform_remote_state.account.outputs.vpc_id
  target_type          = "ip"
  deregistration_delay = 300

  health_check {
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }
}

resource "aws_lb_listener" "pl_listener" {
  load_balancer_arn = aws_lb.pl_nlb.arn
  port              = "80"
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.pl_target_group.arn
  }
}

resource "aws_vpc_endpoint_service" "vpc_endpoint_service" {
  acceptance_required        = false
  network_load_balancer_arns = [aws_lb.pl_nlb.arn]

  allowed_principals = var.vpc_endpoint_service_allowed_principals
}