package tmi

import (
	"strconv"
	"strings"

	tmi "code.justin.tv/chat/tmi/client"
	"code.justin.tv/foundation/twitchclient"
	"github.com/pkg/errors"
	"golang.org/x/net/context"
)

const (
	giftPaidUpgradeText                      = "{{recipient}} is continuing the Gift Sub they got from {{gifter}}!"
	giftPaidUpgradeTextWithPromo             = "{{recipient}} is continuing the Gift Sub they got from {{gifter}}! They're one of {{promo_gift_count}} gift subs to continue this {{promo_name}}."
	giftPaidUpgradeReplacementRecipient      = "{{recipient}}"
	giftPaidUpgradeReplacementGifter         = "{{gifter}}"
	giftPaidUpgradeReplacementPromoGiftCount = "{{promo_gift_count}}"
	giftPaidUpgradeReplacementPromoName      = "{{promo_name}}"

	keyPromoGiftTotal = "promo-gift-total"
	keyPromoName      = "promo-name"
	keyGifterLogin    = "sender-login"
	keyGifterName     = "sender-name"

	giftPaidUpgradeAnonymousDisplayName = "an anonymous user"
)

type GiftPaidUpgradeUserNoticeParams struct {
	ChannelID      string
	PromoGiftTotal int64
	PromoName      string
	GifterName     string
	GifterLogin    string
	RecipientID    string
	RecipientName  string
	AnonGift       bool
}

func (t *client) SendGiftPaidUpgradeUserNotice(ctx context.Context, params GiftPaidUpgradeUserNoticeParams, reqOpts *twitchclient.ReqOpts) error {
	// Create message params to be sent in user notice
	recipientID, err := strconv.Atoi(params.RecipientID)
	if err != nil {
		return errors.Wrapf(err, "converting recipientID %s to int", params.RecipientID)
	}
	channelID, err := strconv.Atoi(params.ChannelID)
	if err != nil {
		return errors.Wrapf(err, "converting channelID %s to int", params.ChannelID)
	}

	msgParams := buildGiftPaidUpgradeMsgParams(params)

	// Create default system body
	var systemBody string

	if params.PromoName == "" {
		// PromoName is empty so do not add promotion information to systemBody
		systemBody = giftPaidUpgradeText
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementRecipient, params.RecipientName, -1)
	} else {
		// PromoName is populated so add promotion information to systemBody
		systemBody = giftPaidUpgradeTextWithPromo
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementRecipient, params.RecipientName, -1)
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementPromoGiftCount, strconv.FormatInt(params.PromoGiftTotal, 10), -1)
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementPromoName, params.PromoName, -1)
	}

	// gifter of user notice will be the channel if anonGift is enabled
	var msgID MsgID
	if params.AnonGift {
		msgID = AnonGiftPaidUpgrade
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementGifter, giftPaidUpgradeAnonymousDisplayName, -1)
	} else {
		msgID = GiftPaidUpgrade
		systemBody = strings.Replace(systemBody, giftPaidUpgradeReplacementGifter, params.GifterName, -1)
	}

	// Send the user notice
	err = t.SendUserNotice(ctx, tmi.SendUserNoticeParams{
		SenderUserID:      recipientID,
		TargetChannelID:   channelID,
		MsgID:             msgID.String(),
		MsgParams:         msgParams,
		DefaultSystemBody: systemBody,
	}, reqOpts)

	if err != nil {
		return errors.Wrap(err, "failed to send user notice")
	}

	return nil
}

func buildGiftPaidUpgradeMsgParams(params GiftPaidUpgradeUserNoticeParams) []tmi.UserNoticeMsgParam {
	msgParams := []tmi.UserNoticeMsgParam{}

	// only send gifter parameters if gift isn't anonymous
	if !params.AnonGift {
		msgParams = append(msgParams,
			tmi.UserNoticeMsgParam{Key: keyGifterLogin, Value: params.GifterLogin},
			tmi.UserNoticeMsgParam{Key: keyGifterName, Value: params.GifterName},
		)
	}

	// If there is an active promotion with a name, send the promotion stats along with the user notice.
	if params.PromoName != "" {
		msgParams = append(msgParams,
			tmi.UserNoticeMsgParam{Key: keyPromoName, Value: params.PromoName},
			tmi.UserNoticeMsgParam{Key: keyPromoGiftTotal, Value: strconv.FormatInt(params.PromoGiftTotal, 10)},
		)
	}

	return msgParams
}
