package tmi

import (
	"fmt"
	"strconv"
	"strings"

	tmi "code.justin.tv/chat/tmi/client"
	log "code.justin.tv/commerce/logrus"
	"github.com/pkg/errors"
	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
)

const (
	// UserNotice MsgParam Keys
	massGiftCountMsgKey = "mass-gift-count"

	// System message string replacement keys
	systemMessageReplacementChannel       = "{channel}"
	systemMessageReplacementMassGiftCount = "{massGiftCount}"

	subKindMMG1000 = "1000"
	subKindMMG2000 = "2000"
	subKindMMG3000 = "3000"

	mmgStartAnonymousDisplayName = "An anonymous user"
)

var (
	mmgSystemMessageMap = map[giftCountBucket]map[string]string{
		// Used when gifter wants to hide their gift count
		giftCountBucketZero: {
			subKindMMG1000: "{name} is gifting {massGiftCount} Tier 1 Subs to {channel}'s community!",
			subKindMMG2000: "{name} is gifting {massGiftCount} Tier 2 Subs to {channel}'s community!",
			subKindMMG3000: "{name} is gifting {massGiftCount} Tier 3 Subs to {channel}'s community!",
		},
		// Normal Gift Message
		giftCountBucketMoreThanOne: {
			subKindMMG1000: "{name} is gifting {massGiftCount} Tier 1 Subs to {channel}'s community! They've gifted a total of {count} in the channel!",
			subKindMMG2000: "{name} is gifting {massGiftCount} Tier 2 Subs to {channel}'s community! They've gifted a total of {count} in the channel!",
			subKindMMG3000: "{name} is gifting {massGiftCount} Tier 3 Subs to {channel}'s community! They've gifted a total of {count} in the channel!",
		},
	}
)

//MMGStartUserNoticeParams contains the params necessary to construct and send a mmg start UserNotice message
type MMGStartUserNoticeParams struct {
	SenderUserID       string
	TargetChannelID    string
	OriginID           string
	SenderDisplayName  string
	ChannelDisplayName string
	SubKind            string
	SubPlan            string
	SenderGiftCount    int
	MassGiftCount      int
	AnonGift           bool
}

func (t *client) SendMMGStartChatUserNotice(ctx context.Context, params MMGStartUserNoticeParams, reqOpts *twitchclient.ReqOpts) error {
	senderUserID, err := strconv.Atoi(params.SenderUserID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubscriptionUserNotice failed to convert SenderID %s to int", params.SenderUserID)
		log.WithError(err).Error(err)
		return err
	}

	targetChannelID, err := strconv.Atoi(params.TargetChannelID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubscriptionUserNotice failed to convert TargetID %s to int", params.TargetChannelID)
		log.WithError(err).Error(err)
		return err
	}

	var giftSystemMessageBucket giftCountBucket
	if params.AnonGift || params.SenderGiftCount == 0 {
		giftSystemMessageBucket = giftCountBucketZero
	} else {
		giftSystemMessageBucket = giftCountBucketMoreThanOne
	}

	systemMessage, ok := mmgSystemMessageMap[giftSystemMessageBucket][params.SubKind]
	if !ok {
		err = errors.New(fmt.Sprintf("TMI: SendMMGStartChatUserNotice could not find a SystemMessage for SubPlan of %s", params.SubPlan))
		log.WithError(err).Error(err)
		return err
	}

	// sender of user notice will be the channel if anonGift is enabled
	var msgID MsgID
	if params.AnonGift {
		msgID = MassMysteryGift
		systemMessage = strings.Replace(systemMessage, systemMessageReplacementName, mmgStartAnonymousDisplayName, -1)
	} else {
		msgID = MassMysteryGift
		systemMessage = strings.Replace(systemMessage, systemMessageReplacementName, params.SenderDisplayName, -1)
	}
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementChannel, params.ChannelDisplayName, -1)
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementCount, strconv.Itoa(params.SenderGiftCount), -1)
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementMassGiftCount, strconv.Itoa(params.MassGiftCount), -1)

	msgParams := []tmi.UserNoticeMsgParam{
		{Key: subPlanMsgKey, Value: params.SubPlan},
		{Key: massGiftCountMsgKey, Value: strconv.Itoa(params.MassGiftCount)},
		{Key: originIDMsgKey, Value: hashString(params.OriginID)},
	}

	// don't expose sender count
	if !params.AnonGift {
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: giftSenderCountMsgKey, Value: strconv.Itoa(params.SenderGiftCount)})
	}

	userNoticeParams := tmi.SendUserNoticeParams{
		DefaultSystemBody: systemMessage,
		MsgID:             msgID.String(),
		MsgParams:         msgParams,
		SenderUserID:      senderUserID,
		TargetChannelID:   targetChannelID,
	}

	err = t.SendUserNotice(ctx, userNoticeParams, nil)
	if err != nil {
		err = errors.Wrap(err, "TMI: Error while sending MMG UserNotice")
		log.WithError(err).Error(err)
		return err
	}

	return nil
}
