package tmi

import (
	"crypto/sha1"
	"fmt"
	"io"
	"strconv"
	"strings"

	tmi "code.justin.tv/chat/tmi/client"
	log "code.justin.tv/commerce/logrus"
	"github.com/pkg/errors"
	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
)

type giftCountBucket int

const (
	giftCountBucketZero giftCountBucket = iota
	giftCountBucketOne
	giftCountBucketMoreThanOne
)

type subMsgBucket int

const (
	shareStreakAndCumulativeTenure subMsgBucket = iota
	shareNoTenureNew
	shareCumulativeTenure
	shareStreakTenure
	shareNoTenure
)

const (
	// UserNotice MsgParam Keys
	subPlanMsgKey                  = "sub-plan"
	subNameMsgKey                  = "sub-plan-name"
	monthsMsgKey                   = "months"
	streakTenureMsgKey             = "streak-tenure-months"
	cumulativeTenureMsgKey         = "cumulative-tenure-months"
	shareStreakTenureKey           = "should-share-streak-tenure"
	giftRecipientIDMsgKey          = "recipient-id"
	giftRecipientNameMsgKey        = "recipient-user-name"
	giftRecipientDisplayNameMsgKey = "recipient-display-name"
	giftSenderCountMsgKey          = "sender-count"
	funString                      = "fun-string"
	originIDMsgKey                 = "origin-id"

	// Names used by Rails Internally
	subKindPrime      = "premium"
	subKindCustom     = "subscription"
	subKindCustomGift = "Custom"
	subKind1000       = "1000"
	subKind2000       = "2000"
	subKind3000       = "3000"

	// System message string replacement keys
	systemMessageReplacementName             = "{name}"
	systemMessageReplacementRecipient        = "{recipient}"
	systemMessageReplacementCumulativeMonths = "{cumulative}"
	systemMessageReplacementStreakMonths     = "{streak}"
	systemMessageReplacementCount            = "{count}"

	// Backup default fields for TMI msg-params
	defaultProductCustomName = "Channel Subscription"

	sendSubscriptionAnonymousDisplayName = "An anonymous user"
)

var (
	// Contains all variations of UserNotice system messages
	subSystemMessageMap = map[subMsgBucket]map[string]string{
		shareStreakAndCumulativeTenure: {
			subKindPrime:  "{name} Subscribed with Twitch Prime. They subscribed for {cumulative} months, currently on a {streak} month streak!",
			subKindCustom: "{name} Subscribed for {cumulative} months, currently on a {streak} month streak!",
			subKind1000:   "{name} Subscribed at Tier 1. They subscribed for {cumulative} months, currently on a {streak} month streak!",
			subKind2000:   "{name} Subscribed at Tier 2. They subscribed for {cumulative} months, currently on a {streak} month streak!",
			subKind3000:   "{name} Subscribed at Tier 3. They subscribed for {cumulative} months, currently on a {streak} month streak!",
		},
		shareCumulativeTenure: {
			subKindPrime:  "{name} Subscribed with Twitch Prime. They subscribed for {cumulative} months!",
			subKindCustom: "{name} Subscribed. They subscribed for {cumulative} months!",
			subKind1000:   "{name} Subscribed at Tier 1. They subscribed for {cumulative} months!",
			subKind2000:   "{name} Subscribed at Tier 2. They subscribed for {cumulative} months!",
			subKind3000:   "{name} Subscribed at Tier 3. They subscribed for {cumulative} months!",
		},
		shareNoTenureNew: {
			subKindPrime:  "{name} Subscribed with Twitch Prime.",
			subKindCustom: "{name} Subscribed.",
			subKind1000:   "{name} Subscribed at Tier 1.",
			subKind2000:   "{name} Subscribed at Tier 2.",
			subKind3000:   "{name} Subscribed at Tier 3.",
		},
		shareStreakTenure: {
			subKindPrime:  "{name} just subscribed with Twitch Prime. {name} subscribed for {streak} months in a row!",
			subKindCustom: "{name} subscribed for {streak} months in a row!",
			subKind1000:   "{name} just subscribed with a Tier 1 sub. {name} subscribed for {streak} months in a row!",
			subKind2000:   "{name} just subscribed with a Tier 2 sub. {name} subscribed for {streak} months in a row!",
			subKind3000:   "{name} just subscribed with a Tier 3 sub. {name} subscribed for {streak} months in a row!",
		},
		shareNoTenure: {
			subKindPrime:  "{name} just subscribed with Twitch Prime!",
			subKindCustom: "{name} just subscribed!",
			subKind1000:   "{name} just subscribed with a Tier 1 sub!",
			subKind2000:   "{name} just subscribed with a Tier 2 sub!",
			subKind3000:   "{name} just subscribed with a Tier 3 sub!",
		},
	}

	giftSystemMessageMap = map[giftCountBucket]map[string]string{
		// Legacy message. Used for the broadcaster themself or in cases where pantheon is unavailable.
		giftCountBucketZero: {
			subKindCustomGift: "{name} gifted a subscription to {recipient}!",
			subKind1000:       "{name} gifted a Tier 1 sub to {recipient}!",
			subKind2000:       "{name} gifted a Tier 2 sub to {recipient}!",
			subKind3000:       "{name} gifted a Tier 3 sub to {recipient}!",
		},
		// First time Gift message
		giftCountBucketOne: {
			subKindCustomGift: "{name} gifted a subscription to {recipient}! This is their first Gift Sub in the channel!",
			subKind1000:       "{name} gifted a Tier 1 sub to {recipient}! This is their first Gift Sub in the channel!",
			subKind2000:       "{name} gifted a Tier 2 sub to {recipient}! This is their first Gift Sub in the channel!",
			subKind3000:       "{name} gifted a Tier 3 sub to {recipient}! This is their first Gift Sub in the channel!",
		},
		// Normal Gift Message
		giftCountBucketMoreThanOne: {
			subKindCustomGift: "{name} gifted a subscription to {recipient}! They have given {count} Gift Subs in the channel!",
			subKind1000:       "{name} gifted a Tier 1 sub to {recipient}! They have given {count} Gift Subs in the channel!",
			subKind2000:       "{name} gifted a Tier 2 sub to {recipient}! They have given {count} Gift Subs in the channel!",
			subKind3000:       "{name} gifted a Tier 3 sub to {recipient}! They have given {count} Gift Subs in the channel!",
		},
	}
)

// SubscriptionUserNoticeParams contains the params necessary to construct and send a subscription UserNotice message
type SubscriptionUserNoticeParams struct {
	SenderUserID           string
	TargetChannelID        string
	SenderDisplayName      string
	CustomMessage          string
	MsgID                  MsgID
	SubKind                string
	SubPlan                string
	SubCustomName          string
	ShouldShareStreak      bool
	CumulativeTenureMonths int
	StreakTenureMonths     int
}

func (t *client) SendSubscriptionUserNotice(ctx context.Context, params SubscriptionUserNoticeParams, reqOpts *twitchclient.ReqOpts) error {
	senderUserID, err := strconv.Atoi(params.SenderUserID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubscriptionUserNotice failed to convert SenderID %s to int", params.SenderUserID)
		log.WithError(err).Error(err)
		return err
	}

	targetChannelID, err := strconv.Atoi(params.TargetChannelID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubscriptionUserNotice failed to convert TargetID %s to int", params.TargetChannelID)
		log.WithError(err).Error(err)
		return err
	}

	isCumulativeTenureEnabled := t.CumulativeTenureConfig.IsCumulativeTenureEnabled(params.SenderUserID)

	var systemMessage string
	var ok bool
	if isCumulativeTenureEnabled {
		if params.StreakTenureMonths < 2 {
			systemMessage, ok = subSystemMessageMap[shareNoTenureNew][params.SubKind]
		} else if params.ShouldShareStreak {
			systemMessage, ok = subSystemMessageMap[shareStreakAndCumulativeTenure][params.SubKind]
		} else {
			systemMessage, ok = subSystemMessageMap[shareCumulativeTenure][params.SubKind]
		}
	} else {
		if params.StreakTenureMonths < 2 {
			systemMessage, ok = subSystemMessageMap[shareNoTenure][params.SubKind]
		} else {
			systemMessage, ok = subSystemMessageMap[shareStreakTenure][params.SubKind]
		}
	}
	if !ok {
		err = errors.Wrapf(err, "TMI: SendSubscriptionUserNotice could not find a SystemMessage for SubKind of %s", params.SubKind)
		log.WithError(err).Error(err)
		return err
	}

	systemMessage = strings.Replace(systemMessage, systemMessageReplacementName, params.SenderDisplayName, -1)
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementCumulativeMonths, strconv.Itoa(params.CumulativeTenureMonths), -1)
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementStreakMonths, strconv.Itoa(params.StreakTenureMonths), -1)

	customName := params.SubCustomName
	if len(customName) == 0 {
		customName = defaultProductCustomName
	}

	msgParams := []tmi.UserNoticeMsgParam{
		{Key: subPlanMsgKey, Value: params.SubPlan},
		{Key: subNameMsgKey, Value: customName},
	}

	if isCumulativeTenureEnabled {
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: monthsMsgKey, Value: "0"})
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: cumulativeTenureMsgKey, Value: strconv.Itoa(params.CumulativeTenureMonths)})
		if params.ShouldShareStreak {
			msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: streakTenureMsgKey, Value: strconv.Itoa(params.StreakTenureMonths)})
		}

		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: shareStreakTenureKey, Value: strconv.FormatBool(params.ShouldShareStreak)})
	} else {
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: monthsMsgKey, Value: strconv.Itoa(params.StreakTenureMonths)})
	}

	userNoticeParams := tmi.SendUserNoticeParams{
		Body:              params.CustomMessage,
		DefaultSystemBody: systemMessage,
		MsgID:             params.MsgID.String(),
		MsgParams:         msgParams,
		SenderUserID:      senderUserID,
		TargetChannelID:   targetChannelID,
	}

	err = t.SendUserNotice(ctx, userNoticeParams, nil)
	if err != nil {
		err = errors.Wrap(err, "TMI: Error while sending subscription UserNotice")
		log.WithError(err).Error(err)
		return err
	}

	return nil
}

// SubGiftUserNoticeParams contains the params necessary to construct and send a gift UserNotice message
type SubGiftUserNoticeParams struct {
	SenderUserID         string
	RecipientUserID      string
	TargetChannelID      string
	OriginID             string
	SenderDisplayName    string
	RecipientDisplayName string
	RecipientLogin       string
	SubKind              string
	SubPlan              string
	SubCustomName        string
	TenureMonths         int
	SenderGiftCount      int
	AnonGift             bool
}

func (t *client) SendSubGiftUserNotice(ctx context.Context, params SubGiftUserNoticeParams, reqOpts *twitchclient.ReqOpts) error {
	return t.sendSubGiftUserNotice(ctx, params, reqOpts, getFunString)
}

type getFunStringFn func() (string, FunStringType)

func (t *client) sendSubGiftUserNotice(ctx context.Context, params SubGiftUserNoticeParams, reqOpts *twitchclient.ReqOpts, getFunStringFn getFunStringFn) error {
	senderUserID, err := strconv.Atoi(params.SenderUserID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubGiftUserNotice failed to convert SenderID %s to int", params.SenderUserID)
		log.WithError(err).Error(err)
		return err
	}

	targetChannelID, err := strconv.Atoi(params.TargetChannelID)
	if err != nil {
		err = errors.Wrapf(err, "TMI: SendSubGiftUserNotice failed to convert TargetID %s to int", params.TargetChannelID)
		log.WithError(err).Error(err)
		return err
	}

	customName := params.SubCustomName
	if len(customName) == 0 {
		customName = defaultProductCustomName
	}

	var giftSystemMessageBucket giftCountBucket
	if params.AnonGift || params.SenderGiftCount == 0 {
		giftSystemMessageBucket = giftCountBucketZero
	} else if params.SenderGiftCount == 1 {
		giftSystemMessageBucket = giftCountBucketOne
	} else {
		giftSystemMessageBucket = giftCountBucketMoreThanOne
	}

	// get funString, may or may not be used
	funString, funStringType := getFunStringFn()

	// build msg params
	msgParams := buildMsgParams(params, customName, funStringType)

	systemMessage, ok := giftSystemMessageMap[giftSystemMessageBucket][params.SubKind]
	if !ok {
		err = errors.New(fmt.Sprintf("TMI: SendSubGiftUserNotice could not find a SystemMessage for SubPlan of %s", params.SubPlan))
		log.WithError(err).Error(err)
		return err
	}

	// sender of user notice will be the channel if anonGift is enabled
	var msgID MsgID
	if params.AnonGift {
		msgID = Gift
		systemMessage = strings.Replace(systemMessage, systemMessageReplacementName, sendSubscriptionAnonymousDisplayName, -1)
		// add a fun string for anon gifts
		systemMessage = strings.Join([]string{systemMessage, funString}, " ")
	} else {
		msgID = Gift
		systemMessage = strings.Replace(systemMessage, systemMessageReplacementName, params.SenderDisplayName, -1)
	}
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementRecipient, params.RecipientDisplayName, -1)
	systemMessage = strings.Replace(systemMessage, systemMessageReplacementCount, strconv.Itoa(params.SenderGiftCount), -1)

	userNoticeParams := tmi.SendUserNoticeParams{
		DefaultSystemBody: systemMessage,
		MsgID:             msgID.String(),
		MsgParams:         msgParams,
		SenderUserID:      senderUserID,
		TargetChannelID:   targetChannelID,
	}

	err = t.SendUserNotice(ctx, userNoticeParams, nil)
	if err != nil {
		err = errors.Wrap(err, "TMI: Error while sending sub gift UserNotice")
		log.WithError(err).Error(err)
		return err
	}

	return nil
}

func buildMsgParams(params SubGiftUserNoticeParams, customName string, funStringType FunStringType) []tmi.UserNoticeMsgParam {
	msgParams := []tmi.UserNoticeMsgParam{
		{Key: monthsMsgKey, Value: strconv.Itoa(params.TenureMonths)},
		{Key: subPlanMsgKey, Value: params.SubPlan},
		{Key: subNameMsgKey, Value: customName},
		{Key: giftRecipientIDMsgKey, Value: params.RecipientUserID},
		{Key: giftRecipientNameMsgKey, Value: params.RecipientLogin},
		{Key: giftRecipientDisplayNameMsgKey, Value: params.RecipientDisplayName},
		{Key: originIDMsgKey, Value: hashString(params.OriginID)},
	}

	// only expose sender gift count if anonGift disabled
	if !params.AnonGift {
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: giftSenderCountMsgKey, Value: strconv.Itoa(params.SenderGiftCount)})
	}

	// additional UserNoticeMsgParams passed for anonGifting
	if params.AnonGift {
		msgParams = append(msgParams, tmi.UserNoticeMsgParam{Key: funString, Value: funStringType.String()})
	}

	return msgParams
}

func hashString(str string) string {
	h := sha1.New()

	_, err := io.WriteString(h, str)
	if err != nil {
		return str
	}

	return fmt.Sprintf("% x", h.Sum(nil))
}
