import { AccountStack } from './account-stack/index';
import { ECRStack } from './ecr-stack/index';
import { VPCEndpointsStack } from './vpc-endpoints-stack/index';
import { InfraStack } from './infra-stack/index';
import { ServiceStack } from './service-stack/index';
import { DynamoDBStack } from './dynamodb-stack/index';

import cdk = require('@aws-cdk/core');

const AWS_ID_PROD = '774733389815';
const AWS_ID_DEV = '454314487337';
const REGION = 'us-west-2';
const VPC_NAME_PROD = 'twitch-secretshop-prod';
const VPC_NAME_DEV = 'twitch-secretshop-dev';

function buildProd(app: cdk.App) {
  const env = {
    account: AWS_ID_PROD,
    region: REGION,
  };

  new AccountStack(app, 'TwitchAccount', { env });
  const ecrStack = new ECRStack(app, 'SecretShopECR', { env });
  new InfraStack(app, 'Infra', { env });

  new VPCEndpointsStack(app, 'VPCEndpoints', {
    env,
    services: [
      {
        name: 'ripley',
        endpointName: 'com.amazonaws.vpce.us-west-2.vpce-svc-0e9e8a2896b1d5e89',
        port: 80,
        domainName: 'ripley.internal.justin.tv',
      },
    ],
    vpcName: VPC_NAME_PROD,
    ldapVPCEndpoint: 'vpce-01d852ad0d301da26-7yebyqx1.vpce-svc-02673a2089b1caf6d.us-west-2.vpce.amazonaws.com',
    ldapRecordName: 'ldap-vpce-01d852ad0d301da26.internal.justin.tv',
  });

  new ServiceStack(app, 'Service', {
    env,
    environment: 'production',
    ecr: ecrStack.repository,
    vpcName: VPC_NAME_PROD,
    whitelistedPrincipals: [
      'arn:aws:iam::989470033077:root', // twitch-cb-aws
      'arn:aws:iam::787149559823:root', // twitch-graphql-prod
    ],
    includeCanary: true,
  });

  new DynamoDBStack(app, 'DynamoDB', {
    env,
    awsAccountID: AWS_ID_PROD,
  });
}

function buildDev(app: cdk.App) {
  const env = {
    account: AWS_ID_DEV,
    region: REGION,
  };

  new AccountStack(app, 'TwitchAccount', { env });
  const ecrStack = new ECRStack(app, 'SecretShopECR', { env });
  new InfraStack(app, 'Infra', { env });

  new VPCEndpointsStack(app, 'VPCEndpoints', {
    env,
    services: [
      {
        name: 'hallpass',
        endpointName: 'com.amazonaws.vpce.us-west-2.vpce-svc-05c9a9682cec9fa80',
        port: 80,
        domainName: 'staging.hallpass.cb.twitch.a2z.com',
      }, {
        name: 'ripley',
        endpointName: 'com.amazonaws.vpce.us-west-2.vpce-svc-0302ab0bc9133e1a4',
        port: 80,
        domainName: 'ripley.dev.us-west2.internal.justin.tv',
      },
    ],
    vpcName: VPC_NAME_DEV,
    ldapVPCEndpoint: 'vpce-0f09b18e374981d6a-a91n2n87.vpce-svc-02673a2089b1caf6d.us-west-2.vpce.amazonaws.com',
    ldapRecordName: 'ldap-vpce-0f09b18e374981d6a.internal.justin.tv',
  });

  new ServiceStack(app, 'Service', {
    env,
    environment: 'staging',
    ecr: ecrStack.repository,
    vpcName: VPC_NAME_DEV,
    whitelistedPrincipals: [
      'arn:aws:iam::989470033077:root', // twitch-cb-aws
      'arn:aws:iam::645130450452:root', // twitch-graphql-dev
    ],
    includeCanary: false,
  });

  new DynamoDBStack(app, 'DynamoDB', {
    env,
    awsAccountID: AWS_ID_DEV,
  });
}

const app = new cdk.App();
if (app.node.tryGetContext('stage') == 'prod') {
  buildProd(app);
} else {
  buildDev(app);
}
