// Adapted from https://github.com/aws/aws-cdk/blob/v1.23.0/packages/@aws-cdk/aws-ecs-patterns/lib/ecs/application-load-balanced-ecs-service.ts
import { Ec2Service, Ec2TaskDefinition } from '@aws-cdk/aws-ecs';
import { Construct } from '@aws-cdk/core';
import { IVpc } from '@aws-cdk/aws-ec2';
import { ContainerImage, ICluster, LogDriver } from '@aws-cdk/aws-ecs';
import { ApplicationTargetGroup } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IRole } from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';

export interface ApplicationLoadBalancedEc2ServiceProps {
  readonly cluster: ICluster;
  readonly vpc?: IVpc;
  readonly taskImageOptions: ApplicationLoadBalancedTaskImageOptions;
  readonly desiredCount: number;
  readonly serviceName?: string;
  readonly healthCheckGracePeriod?: cdk.Duration;
  readonly maxHealthyPercent?: number;
  readonly minHealthyPercent?: number;
  readonly targetGroup: ApplicationTargetGroup;
  readonly cpu?: number;
  readonly memoryLimitMiB?: number;
  readonly memoryReservationMiB?: number;
}

export interface ApplicationLoadBalancedTaskImageOptions {
  readonly image: ContainerImage;
  readonly environment?: { [key: string]: string };
  readonly logDriver: LogDriver;
  readonly executionRole?: IRole;
  readonly taskRole?: IRole;
  readonly containerName?: string;
  readonly containerPort?: number;
  readonly family?: string;
}

export class ApplicationLoadBalancedEc2Service extends cdk.Construct {
  public readonly taskDefinition: Ec2TaskDefinition;
  constructor(scope: Construct, id: string, props: ApplicationLoadBalancedEc2ServiceProps) {
    super(scope, id);
    const taskImageOptions = props.taskImageOptions;
    this.taskDefinition = new Ec2TaskDefinition(this, 'TaskDef', {
      executionRole: taskImageOptions.executionRole,
      taskRole: taskImageOptions.taskRole,
      family: taskImageOptions.family,
    });

    const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
    const container = this.taskDefinition.addContainer(containerName, {
      image: taskImageOptions.image,
      cpu: props.cpu,
      memoryLimitMiB: props.memoryLimitMiB,
      memoryReservationMiB: props.memoryReservationMiB,
      environment: taskImageOptions.environment,
      logging: taskImageOptions.logDriver,
    });
    container.addPortMappings({
      containerPort: taskImageOptions.containerPort || 80,
    });

    const service = new Ec2Service(this, "Service", {
      cluster: props.cluster,
      desiredCount: props.desiredCount,
      taskDefinition: this.taskDefinition,
      assignPublicIp: false,
      serviceName: props.serviceName,
      healthCheckGracePeriod: props.healthCheckGracePeriod,
      minHealthyPercent: props.minHealthyPercent,
      maxHealthyPercent: props.maxHealthyPercent,
    });

    props.targetGroup.addTarget(service);
  }
}
