import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as eslb from '@twitch-easymode/endpoint-service-loadbalancer';
import * as elb from '@aws-cdk/aws-elasticloadbalancingv2';
import * as logs from '@aws-cdk/aws-logs';
import * as iam from '@aws-cdk/aws-iam';
import * as route53 from '@aws-cdk/aws-route53';
import * as alias from '@aws-cdk/aws-route53-targets';
import * as cdk from '@aws-cdk/core';
import { ApplicationLoadBalancedEc2Service } from './application-service';

interface ServiceStackProps extends cdk.StackProps {
  ecr: ecr.IRepository;
  environment: string;
  vpcName: string;
  whitelistedPrincipals?: string[];
  includeCanary: boolean;
}

interface ECSServiceProps extends ServiceStackProps {
  clusterName: string;
  listenerPort: number;
  serviceName: string;
  maxCapacity: number;
  minCapacity: number;
}

export class ServiceStack extends cdk.Stack {
  private vpc: ec2.IVpc;
  private listener: elb.ApplicationListener;
  private targetGroups: elb.ApplicationTargetGroup[] = [];

  private createECSService(id: string, props: ECSServiceProps) {
    const targetGroup = this.listener.addTargets(id + 'ECS', {
      port: props.listenerPort,
    });
    targetGroup.configureHealthCheck({
      path: '/health',
      port: 'traffic-port',
    });
    this.targetGroups.push(targetGroup);

    // Creates task definition and ECS / EC2 cluster
    const cluster = new ecs.Cluster(this, id + 'Cluster', {
      vpc: this.vpc,
      clusterName: props.clusterName,
    });

    const asg = cluster.addCapacity(id + 'ASG', {
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
      maxCapacity: props.maxCapacity,
      minCapacity: props.minCapacity,
      machineImage: new ecs.EcsOptimizedAmi(),
    });
    asg.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
    asg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));

    asg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));

    const ec2Service = new ApplicationLoadBalancedEc2Service(this, id + 'ECS', {
      cluster,
      memoryLimitMiB: 2048,
      desiredCount: 2,
      targetGroup,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
      serviceName: props.serviceName,
      taskImageOptions: {
        containerPort: 8080,
        image: ecs.ContainerImage.fromEcrRepository(props.ecr),
        family: `${props.serviceName}-TaskDefinition`,
        logDriver: new ecs.AwsLogDriver({
          logGroup: new logs.LogGroup(this, id + 'WebLogGroup'),
          streamPrefix: props.serviceName,
        }),
        environment: {
          ENVIRONMENT: props.environment,
        },
      },
    });

    ec2Service.taskDefinition.taskRole.addManagedPolicy(
      iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonDynamoDBFullAccess')
    );

    return ec2Service;
  }

  constructor(scope: cdk.App, id: string, props: ServiceStackProps) {
    super(scope, id, props);

    this.vpc = ec2.Vpc.fromLookup(this, id, {
      vpcName: props.vpcName,
    });

    const loadBalancer = new elb.ApplicationLoadBalancer(this, 'ECSLB', {
      vpc: this.vpc,
      internetFacing: false,
    });

    const listenerPort = 80;
    this.listener = loadBalancer.addListener('PublicListener', {
      protocol: elb.ApplicationProtocol.HTTP,
      port: listenerPort,
      open: true
    });

    const service = this.createECSService('', {
      ...props,
      clusterName: 'SecretShop',
      listenerPort,
      serviceName: 'SecretShop',
      maxCapacity: 4,
      minCapacity: 2,
    })

    if (props.includeCanary) {
      this.createECSService('Canary', {
        ...props,
        clusterName: 'SecretShop-canary',
        listenerPort,
        serviceName: 'SecretShop-canary',
        maxCapacity: 1,
        minCapacity: 1,
      })

      // Update listener's default rule to forward a percentage of traffic to canary
      const cfnListener = this.listener.node.defaultChild as elb.CfnListener;
      cfnListener.addPropertyOverride('DefaultActions', [
        {
          Type: 'forward',
          ForwardConfig: {
            TargetGroups: [
              { 
                TargetGroupArn: this.targetGroups[0].targetGroupArn,
                Weight: 90,
              },
              { 
                TargetGroupArn: this.targetGroups[1].targetGroupArn,
                Weight: 10,
              },
            ],
          },
        },
      ]);
    }

    // Creates a NLB and a Lambda to sync with pre-existing ALB.
    const endpointServiceLoadBalancer = new eslb.EndpointServiceLoadBalancer(this, 'EndpointServiceLoadBalancer', {
      protocol: elb.ApplicationProtocol.HTTP,
      alb: loadBalancer,
      albListener: this.listener,
      albListenerPort: 80,
      vpc: this.vpc,
      cidr: this.vpc.vpcCidrBlock,
    });

    // Create a VPC Endpoint service so others can create VPC endpoints to
    const vpces = new ec2.CfnVPCEndpointService(this, 'EndpointService', {
      networkLoadBalancerArns: [endpointServiceLoadBalancer.nlb.loadBalancerArn],
      acceptanceRequired: false,
    });

    if (props.whitelistedPrincipals) {
      // Whitelist accounts that are allowed to create VPC endpoints with our VPC Endpoint service
      new ec2.CfnVPCEndpointServicePermissions(this, 'WhitelistedPrincipals', {
        allowedPrincipals: props.whitelistedPrincipals,
        serviceId: vpces.ref,
      });
    }

    // Creates public hosted zone for twitch.a2z.com subdomain
    const publicHostedZone = new route53.HostedZone(this, 'PublicHostedZone', {
      zoneName: `${props.environment}.secretshop.twitch.a2z.com`, 
      comment: 'Public service resolution for a2z',
    })

    new route53.TxtRecord(this, 'TestRecord',  {
      values: ['If you\'re reading this it works'],
      zone: publicHostedZone, 
      comment: 'Test record', 
      ttl: cdk.Duration.minutes(5), 
      recordName: `_test.${publicHostedZone.zoneName}`,
    })
    
    new route53.ARecord(this, 'DNS', {
      zone: publicHostedZone,
      recordName: `${props.environment}.secretshop.twitch.a2z.com`,
      target: route53.RecordTarget.fromAlias(new alias.LoadBalancerTarget(loadBalancer)),
    });

    // Creates private record for secretshop's twitch.a2z.com subdomain to be accessed in the VPC
    const privateHostedZone = new route53.HostedZone(this, 'PrivateHostedZone', {
      zoneName: `${props.environment}.secretshop.twitch.a2z.com`, 
      comment: 'Private service resolution for SecretShop a2z',
      vpcs: [this.vpc],
    })
    new route53.ARecord(this, 'PrivateAliasRecord', {
      target: route53.RecordTarget.fromAlias(new alias.LoadBalancerTarget(loadBalancer)),
      zone: privateHostedZone, 
      recordName: `${props.environment}.secretshop.twitch.a2z.com`,
    })
  }
}
