import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import * as route53 from '@aws-cdk/aws-route53';
import plEndpoint = require('@twitch-easymode/privatelink-endpoint');

interface ServiceDependency {
  name: string,
  endpointName: string,
  port: number,
  domainName: string,
}

interface VPCEndpointsStackProps extends cdk.StackProps {
  services: ServiceDependency[],
  vpcName: string,
  ldapVPCEndpoint: string,
  ldapRecordName: string,
}

export class VPCEndpointsStack extends cdk.Stack {
  constructor(scope: cdk.Construct, name: string, props: VPCEndpointsStackProps) {
    super(scope, name, props);

    const vpc = ec2.Vpc.fromLookup(this, name, {
      vpcName: props.vpcName,
    });

    for (const service of props.services) {
      // This will create a private route53 zone within your account,
      // accessing domainName will route to the VPC endpoint
      const endpoint = new plEndpoint.PrivateLinkEndpoint(this, service.name, {
        vpc,
        service: {
          name: service.endpointName,
          port: service.port,
        },
        hostedZones: [{
          zoneName: service.domainName,
        }],
      });

      endpoint.connections.allowFromAnyIpv4(new ec2.Port({
        protocol: ec2.Protocol.ALL,
        stringRepresentation: service.name,
      }));
    }

    // Creates private hosted zone for justin.tv
    const ldapHostedZone = new route53.HostedZone(this, 'LDAPHostedZone', {
      zoneName: 'justin.tv', 
      comment: 'Hosted zone that allows SSL communication to services expecting a *.justin.tv endpoint',
      vpcs: [vpc],
    })

    new route53.ARecord(this, 'LDAPRecord', {
      target: asAliasTarget(props.ldapVPCEndpoint),
      zone: ldapHostedZone, 
      comment: 'Alias A record for Twitch LDAP access',
      recordName: props.ldapRecordName,
    })
  }
}

function asAliasTarget(dnsName: string) {
  const aliasTarget = {
    bind: () => ({
      // Hosted zone for all us-west-2 VPC Endpoints
      hostedZoneId: 'Z1YSA3EXCYUU9Z',
      dnsName,
    }),
  };
  return route53.RecordTarget.fromAlias(aliasTarget);
}
