package main

import (
	"fmt"
	"net/http"
	"os"

	"code.justin.tv/cb/secretshop/config"
	"code.justin.tv/cb/secretshop/internal/clients/dynamodb"
	"code.justin.tv/cb/secretshop/internal/clients/ripley"
	"code.justin.tv/cb/secretshop/internal/server"
	"code.justin.tv/cb/secretshop/internal/serverutil"
	"code.justin.tv/cb/secretshop/rpc/secretshop"

	"goji.io/pat"

	log "github.com/sirupsen/logrus"
	goji "goji.io"
)

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("secretshop: no variable 'ENVIRONMENT' set in env")
	}

	config, err := config.New(env)
	if err != nil {
		log.Fatalf("secretshop: failed to load config - %s", err.Error())
	}

	region := config.Region
	if region == "" {
		log.Fatal("secretshop: no region found in config")
	}

	dynamoTableName := config.StoreTableName
	if dynamoTableName == "" {
		log.Fatal("secretshop: no dynamo table name found in config")
	}

	dynamoDBClient, err := dynamodb.NewClient(dynamoTableName, region)
	if err != nil {
		log.Fatal("secretshop: failed to create dynamodb client", err)
	}

	server := &server.Server{
		DynamoDB: dynamoDBClient,
		Ripley:   ripley.NewClient(),
	}
	twirpHandler := secretshop.NewSecretShopServer(server, nil)

	mux := goji.NewMux()
	mux.Handle(pat.Post(secretshop.SecretShopPathPrefix+"*"), twirpHandler)
	mux.HandleFunc(pat.Get("/health"), serverutil.HealthCheck)

	addr := fmt.Sprintf(":%s", config.Port)
	log.Infof("secretshop: server listening on port %s", addr)

	if err := http.ListenAndServe(addr, mux); err != nil {
		log.Fatal("server error: ", err)
	}
}
