package config

import (
	"encoding/json"
	"fmt"
	"io/ioutil"

	log "github.com/sirupsen/logrus"
)

const (
	developmentEnv = "development"
	stagingEnv     = "staging"
	productionEnv  = "production"
)

// Config holds static configuration for secretshop. It does not hold any secrets.
// We need to pass in an environment name to the config loader, which needs to be defined
// as an environment variable when running secretshop.
type Config struct {
	Environment    string `json:"environment"`
	Port           string `json:"port"`
	GlobalsFile    string `json:"globals_file"`
	StoreTableName string `json:"store_table_name"`
	Region         string `json:"region"`
}

func (c *Config) IsDev() bool {
	return c.Environment == developmentEnv
}

func New(env string) (*Config, error) {
	err := validateEnvironment(env)
	if err != nil {
		return nil, err
	}

	c := &Config{
		Environment: env,
	}
	err = c.load(env)
	if err != nil {
		return nil, err
	}

	return c, nil
}

func (c *Config) load(env string) error {
	filePath := fmt.Sprintf("config/%s.json", env)
	log.Infof("loading config from %s", filePath)

	bytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		return err
	}

	return json.Unmarshal(bytes, &c)
}

func validateEnvironment(env string) error {
	if env == "" || (env != productionEnv && env != stagingEnv && env != developmentEnv) {
		return fmt.Errorf("invalid environment '%s' in config loader", env)
	}

	return nil
}
