package dynamodb

import (
	"context"
	"fmt"

	"code.justin.tv/cb/secretshop/internal/quickactions/schema"
	"golang.org/x/sync/errgroup"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

const batchWriteMaxSize = 25

// GetAllStoreItems returns all quick actions scoped to the provided channelID, as well as all global quick actions.
func (c *Client) GetAllStoreItems(ctx context.Context, channelID string) ([]*schema.QuickAction, error) {
	localInput := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("#S = :channelID"),
		ExpressionAttributeNames: map[string]*string{
			"#S": aws.String("scope"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channelID": {S: aws.String(channelID)},
		},
		TableName: aws.String(c.storeTableName),
	}

	globalInput := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("#S = :global"),
		ExpressionAttributeNames: map[string]*string{
			"#S": aws.String("scope"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":global": {S: aws.String(schema.GlobalScope)},
		},
		TableName: aws.String(c.storeTableName),
	}

	eg, egCtx := errgroup.WithContext(ctx)
	var localQuickActions []*schema.QuickAction
	var globalQuickActions []*schema.QuickAction

	eg.Go(func() error {
		localsOutput, err := c.dynamoDB.QueryWithContext(egCtx, localInput)
		if err != nil {
			return err
		}
		err = dynamodbattribute.UnmarshalListOfMaps(localsOutput.Items, &localQuickActions)
		if err != nil {
			return err
		}
		return nil
	})

	eg.Go(func() error {
		globalsOutput, err := c.dynamoDB.QueryWithContext(egCtx, globalInput)
		if err != nil {
			return err
		}
		err = dynamodbattribute.UnmarshalListOfMaps(globalsOutput.Items, &globalQuickActions)
		if err != nil {
			return err
		}
		return nil
	})

	if err := eg.Wait(); err != nil {
		return nil, err
	}

	quickActions := append(globalQuickActions, localQuickActions...)

	return quickActions, nil
}

func (c *Client) PutGlobals(ctx context.Context, globals []*schema.QuickAction) error {
	if len(globals) == 0 {
		return nil
	}

	if len(globals) > batchWriteMaxSize {
		return fmt.Errorf("dynamo: batch size of %d is greater than max allowable (%d)", len(globals), batchWriteMaxSize)
	}

	writes := make([]*dynamodb.WriteRequest, len(globals))
	for i, qa := range globals {
		item, err := dynamodbattribute.MarshalMap(qa)
		if err != nil {
			return err
		}

		writes[i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: item,
			},
		}
	}

	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			c.storeTableName: writes,
		},
	}

	result, err := c.dynamoDB.BatchWriteItemWithContext(ctx, input)
	if err != nil {
		return err
	}

	if len(result.UnprocessedItems) == 0 {
		return nil
	}

	return fmt.Errorf("dynamo: Found %d unprocessed items after batch write", len(result.UnprocessedItems))
}

func (c *Client) AddStoreItem(ctx context.Context, quickAction *schema.QuickAction) error {
	input := &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"name":     {S: aws.String(quickAction.Name)},
			"scope":    {S: aws.String(quickAction.Scope)},
			"category": {S: aws.String(quickAction.Category)},
			"enabled":  {BOOL: aws.Bool(quickAction.Enabled)},
		},
		TableName: &c.storeTableName,
	}

	_, err := c.dynamoDB.PutItemWithContext(ctx, input)
	return err
}

func (c *Client) RemoveItem(ctx context.Context, channelID string, itemName string) (*schema.QuickAction, error) {
	input := &dynamodb.DeleteItemInput{
		TableName: aws.String(c.storeTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"scope": {S: aws.String(channelID)},
			"name":  {S: aws.String(itemName)},
		},
		ReturnValues: aws.String("ALL_OLD"),
	}

	resp, err := c.dynamoDB.DeleteItemWithContext(ctx, input)
	if err != nil {
		return nil, err
	}

	var item *schema.QuickAction
	if err := dynamodbattribute.UnmarshalMap(resp.Attributes, &item); err != nil {
		return nil, err
	}

	return item, err
}

func (c *Client) GetDefaultItems(ctx context.Context, isPartner bool) ([]*schema.QuickAction, error) {
	cond := "is_default_for_all"
	if isPartner {
		cond = "is_default_for_partner"
	}

	input := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("#S = :global"),
		ExpressionAttributeNames: map[string]*string{
			"#S":  aws.String("scope"),
			"#DS": aws.String("default_status"),
			"#DK": aws.String(cond),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":global":  {S: aws.String(schema.GlobalScope)},
			":default": {BOOL: aws.Bool(true)},
		},
		FilterExpression: aws.String("#DS.#DK = :default"),
		TableName:        aws.String(c.storeTableName),
	}

	output, err := c.dynamoDB.QueryWithContext(ctx, input)
	if err != nil {
		return nil, err
	}

	var items []*schema.QuickAction
	if err := dynamodbattribute.UnmarshalListOfMaps(output.Items, &items); err != nil {
		return nil, err
	}
	return items, nil
}
