package server

import (
	"context"

	"code.justin.tv/cb/secretshop/internal/clients/dynamodb"
	"code.justin.tv/cb/secretshop/internal/clients/ripley"
	"code.justin.tv/cb/secretshop/internal/quickactions/conv"
	"code.justin.tv/cb/secretshop/internal/quickactions/schema"

	pb "code.justin.tv/cb/secretshop/rpc/secretshop"
	log "github.com/sirupsen/logrus"
)

type Server struct {
	DynamoDB dynamodb.Datastore
	Ripley   ripley.Payouts
}

func (s *Server) GetQuickActions(ctx context.Context, req *pb.GetQuickActionsReq) (*pb.GetQuickActionsResp, error) {
	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Error("secretshop: GetQuickActions failed - no channelID")
		return nil, ErrMissingChannelID
	}

	quickActions, err := s.DynamoDB.GetAllStoreItems(ctx, channelID)

	if err != nil {
		log.WithError(err).Error("secretshop: GetQuickActions failed to get quick actions")
		return nil, err
	}

	storeActions := conv.DynamoToProtoList(quickActions)

	return &pb.GetQuickActionsResp{
		QuickActions: storeActions,
	}, nil
}

func (s *Server) CreateQuickAction(ctx context.Context, req *pb.CreateQuickActionReq) (*pb.CreateQuickActionResp, error) {
	quickAction := req.GetQuickAction()
	if quickAction == nil {
		log.Error("secretshop: CreateQuickAction failed - no Quick Action")
		return nil, ErrMissingQuickAction
	}

	if quickAction.Scope == schema.GlobalScope {
		return nil, ErrInvalidScope
	}

	dynamoQA := conv.ProtoToDynamoQuickAction(quickAction)

	err := s.DynamoDB.AddStoreItem(ctx, dynamoQA)
	if err != nil {
		return nil, err
	}

	return &pb.CreateQuickActionResp{}, nil
}

func (s *Server) DeleteQuickAction(ctx context.Context, req *pb.DeleteQuickActionReq) (*pb.DeleteQuickActionResp, error) {
	channelID := req.GetChannelID()
	qaName := req.GetName()

	if len(channelID) == 0 {
		return nil, ErrMissingChannelID
	}

	if channelID == schema.GlobalScope {
		return nil, ErrInvalidScope
	}

	if len(qaName) == 0 {
		return nil, ErrMissingName
	}

	removedItem, err := s.DynamoDB.RemoveItem(ctx, channelID, qaName)
	if err != nil {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"qa_name":    qaName,
		}).WithError(err).
			Error("secretshop: failed to remove item from dynamodb")
		return nil, err
	}

	converted := conv.DynamoToProtoQuickAction(removedItem)

	return &pb.DeleteQuickActionResp{
		DeletedItem: converted,
	}, nil
}

func (s *Server) GetDefaultQuickActions(ctx context.Context, req *pb.GetDefaultQuickActionsReq) (*pb.GetDefaultQuickActionsResp, error) {
	channelID := req.GetChannelID()

	if len(channelID) == 0 {
		return nil, ErrMissingChannelID
	}

	isPartner, err := s.Ripley.IsPartner(ctx, channelID)
	if err != nil {
		return nil, err
	}

	defaults, err := s.DynamoDB.GetDefaultItems(ctx, isPartner)
	if err != nil {
		return nil, err
	}

	return &pb.GetDefaultQuickActionsResp{
		QuickActions: conv.DynamoToProtoList(defaults),
	}, nil
}
