package server_test

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/cb/secretshop/internal/quickactions/conv"
	"code.justin.tv/cb/secretshop/internal/quickactions/schema"
	"code.justin.tv/cb/secretshop/internal/server"

	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"

	dynamodb "code.justin.tv/cb/secretshop/internal/mocks/dynamodb"
	pb "code.justin.tv/cb/secretshop/rpc/secretshop"
)

var quickActionList = []*schema.QuickAction{
	{
		Name:     "some-name",
		Scope:    "scope",
		Category: "some-category",
		Enabled:  true,
	},
}

func getTestServer(mockDynamo *dynamodb.MockDatastore) *server.Server {
	return &server.Server{
		DynamoDB: mockDynamo,
	}
}

func TestGetQuickActions(t *testing.T) {
	ctrl := gomock.NewController(t)
	ctx := context.Background()

	t.Run("no channel ID", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.GetQuickActionsReq{
			ChannelID: "",
		}
		mockDB.EXPECT().GetAllStoreItems(gomock.Any(), gomock.Any()).Times(0)

		_, err := s.GetQuickActions(ctx, req)
		assert.Equal(t, server.ErrMissingChannelID, err)
	})

	t.Run("dynamo error", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.GetQuickActionsReq{
			ChannelID: "something",
		}
		mockDB.EXPECT().GetAllStoreItems(gomock.Any(), gomock.Any()).Return(nil, errors.New("error"))

		result, err := s.GetQuickActions(ctx, req)
		assert.NotNil(t, err)
		assert.Nil(t, result)
	})

	t.Run("success", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.GetQuickActionsReq{
			ChannelID: "something",
		}
		mockDB.EXPECT().GetAllStoreItems(gomock.Any(), gomock.Any()).Return(quickActionList, nil)

		expected := conv.DynamoToProtoList(quickActionList)
		result, _ := s.GetQuickActions(ctx, req)
		assert.Equal(t, pb.GetQuickActionsResp{QuickActions: expected}, *result)
	})
}

func TestAddQuickAction(t *testing.T) {
	ctrl := gomock.NewController(t)
	ctx := context.Background()

	t.Run("no quick action", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.CreateQuickActionReq{
			QuickAction: nil,
		}
		mockDB.EXPECT().AddStoreItem(gomock.Any(), gomock.Any()).Times(0)

		_, err := s.CreateQuickAction(ctx, req)
		assert.Equal(t, server.ErrMissingQuickAction, err)
	})

	t.Run("invalid global scope", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.CreateQuickActionReq{
			QuickAction: &pb.QuickAction{
				Name:     "some-name",
				Scope:    "global",
				Category: pb.CategoryType_UNCATEGORIZED,
				Enabled:  true,
			},
		}
		mockDB.EXPECT().AddStoreItem(gomock.Any(), gomock.Any()).Times(0)

		_, err := s.CreateQuickAction(ctx, req)
		assert.Equal(t, server.ErrInvalidScope, err)
	})

	t.Run("dynamo error", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.CreateQuickActionReq{
			QuickAction: &pb.QuickAction{
				Name:     "some-name",
				Scope:    "scope",
				Category: pb.CategoryType_UNCATEGORIZED,
				Enabled:  true,
			},
		}
		mockDB.EXPECT().AddStoreItem(gomock.Any(), gomock.Any()).Return(errors.New("error"))

		res, err := s.CreateQuickAction(ctx, req)
		assert.NotNil(t, err)
		assert.Nil(t, res)
	})

	t.Run("success", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.CreateQuickActionReq{
			QuickAction: &pb.QuickAction{
				Name:     "some-name",
				Scope:    "scope",
				Category: pb.CategoryType_UNCATEGORIZED,
				Enabled:  true,
			},
		}
		mockDB.EXPECT().AddStoreItem(gomock.Any(), gomock.Any()).Return(nil)

		result, _ := s.CreateQuickAction(ctx, req)
		assert.Equal(t, pb.CreateQuickActionResp{}, *result)
	})
}

func TestRemoveQuickAction(t *testing.T) {
	ctrl := gomock.NewController(t)
	ctx := context.Background()

	t.Run("invalid channel id", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.DeleteQuickActionReq{
			ChannelID: "",
		}
		mockDB.EXPECT().RemoveItem(gomock.Any(), gomock.Any(), gomock.Any()).Times(0)

		resp, err := s.DeleteQuickAction(ctx, req)
		assert.Nil(t, resp)
		assert.Equal(t, server.ErrMissingChannelID, err)
	})

	t.Run("invalid name", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.DeleteQuickActionReq{
			ChannelID: "111111",
			Name:      "",
		}
		mockDB.EXPECT().RemoveItem(gomock.Any(), req.ChannelID, gomock.Any()).Times(0)

		resp, err := s.DeleteQuickAction(ctx, req)
		assert.Nil(t, resp)
		assert.Equal(t, server.ErrMissingName, err)
	})

	t.Run("invalid scope", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.DeleteQuickActionReq{
			ChannelID: "global",
			Name:      "qa-name",
		}
		mockDB.EXPECT().RemoveItem(gomock.Any(), req.ChannelID, req.Name).Times(0)

		resp, err := s.DeleteQuickAction(ctx, req)
		assert.Nil(t, resp)
		assert.Equal(t, server.ErrInvalidScope, err)
	})

	t.Run("dynamo error", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.DeleteQuickActionReq{
			ChannelID: "111111",
			Name:      "qa-name",
		}
		mockDB.EXPECT().RemoveItem(gomock.Any(), req.ChannelID, req.Name).Return(nil, errors.New("dynamodb error"))

		res, err := s.DeleteQuickAction(ctx, req)
		assert.NotNil(t, err)
		assert.Nil(t, res)
	})

	t.Run("success", func(t *testing.T) {
		mockDB := dynamodb.NewMockDatastore(ctrl)
		s := getTestServer(mockDB)
		req := &pb.DeleteQuickActionReq{
			ChannelID: "111111",
			Name:      "qa-name",
		}

		ddbExpectedResp := &schema.QuickAction{
			Name:  "qa-name",
			Scope: "111111",
		}
		mockDB.EXPECT().RemoveItem(gomock.Any(), req.ChannelID, req.Name).Return(ddbExpectedResp, nil)

		resp, err := s.DeleteQuickAction(ctx, req)
		assert.Nil(t, err)
		assert.Equal(t, resp.DeletedItem.Name, ddbExpectedResp.Name)
		assert.Equal(t, resp.DeletedItem.Scope, ddbExpectedResp.Scope)
	})
}
