// loadglobals
//
// This script reads in the globals file found at `config/global_quick_actions.json`,
// marshalls the json items into QuickAction structs, and then inserts all of those
// items into the store table, under the `global` scope.
// This should be run anytime there are changes to global actions or new actions are
// added globally.
// This requires proper AWS creds set as env vars in the credential chain. For example, to
// run on production, copy the prod account credentials from isengard to the terminal
// and then run the script in the same terminal.
//
// Usage:
// go run main.go -env <environment>
//
// env can be one of "development" or "production"

package main

import (
	"context"
	"flag"
	"log"
	"math"
	"time"

	"code.justin.tv/cb/secretshop/internal/clients/dynamodb"
	"code.justin.tv/cb/secretshop/internal/quickactions/global"
	"code.justin.tv/cb/secretshop/internal/quickactions/schema"
)

var env = flag.String("env", "", "")

const (
	globalsRelPath     = "../../config/global_quick_actions.json"
	region             = "us-west-2"
	tableName          = "quick-actions-store"
	jobTimeout         = 10 * time.Second
	dynamoMaxBatchSize = 25
)

func init() {
	flag.Parse()
}

func main() {
	start := time.Now()

	if env == nil || len(*env) == 0 {
		usage()
		log.Fatal("❌ globals: failed to parse arguments")
	}

	if len(*env) == 0 {
		usage()
		log.Fatal("❌ globals: failed to parse arguments")
	}

	if *env != "development" && *env != "production" {
		usage()
		log.Fatalf("❌ globals: incorrect environment %s provided", *env)
	}

	log.Printf("globals: loading quick actions from %s and creating dynamo client...", globalsRelPath)
	loadStart := time.Now()

	dynamoDBClient, err := dynamodb.NewClient(tableName, region)
	if err != nil {
		log.Fatal("❌ globals: failed to create dynamodb client", err)
	}

	actions, err := global.LoadGlobals(globalsRelPath)
	if err != nil {
		log.Fatal(err)
	}

	log.Printf("✅ globals: finished loading in %s", time.Since(loadStart))

	batches := float64(len(actions.Globals)) / float64(dynamoMaxBatchSize)
	batchCount := int(math.Ceil(batches))
	log.Printf("globals: found %d items, for a total of %d dynamo batches", len(actions.Globals), batchCount)

	ctx, cancel := context.WithTimeout(context.Background(), jobTimeout)
	defer cancel()

	writeStart := time.Now()
	err = writeGlobals(ctx, actions.Globals, dynamoDBClient, batchCount)
	if err != nil {
		log.Fatal(err)
	}
	log.Printf("✅ globals: finished writing batches to dynamo in %s", time.Since(writeStart))

	select {
	case <-ctx.Done():
		log.Println("❌ global quick action insertion timed out. Some actions may not have been written.")
		log.Fatal(ctx.Err())
	default:
		log.Println("=====================================================")
		log.Println("✅ Global quick action insertion complete!")
		log.Println("⏱  Elapsed: ", time.Since(start))
		log.Println("=====================================================")
	}
}

func writeGlobals(ctx context.Context, globals []*schema.QuickAction, dynamoDBClient *dynamodb.Client, batchCount int) error {
	currBatch := 1
	for idx := 0; idx < len(globals); idx += dynamoMaxBatchSize {
		end := idx + dynamoMaxBatchSize
		if end > len(globals) {
			end = len(globals)
		}

		batch := globals[idx:end]

		log.Printf("globals: writing batch %d of %d batches...", currBatch, batchCount)
		err := dynamoDBClient.PutGlobals(ctx, batch)
		if err != nil {
			return err
		}

		currBatch = currBatch + 1
	}

	return nil
}

func usage() {
	log.Println("usage: go run main.go -env <environment>")
	log.Println("<environment> can be one of 'development' or 'production'")
}
