package main

import (
	"context"
	"fmt"
	"time"

	sqswrapper "code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

var (
	backfillStartTime = time.Date(2018, 4, 1, 00, 0, 0, 0, time.UTC)
	backfillEndTime   = time.Date(2018, 5, 31, 23, 59, 59, 0, time.UTC)
)

var stats = []string{
	// "clips",
	// "sessions",

	// "broadcastidsession",
	// "chatsession",
	// "concurrentssession",
	// "serverfollowsession",
	"subscriptionspurchasesuccessssession",
	// "videoplaygeosession",
	// "videoplayplatformsession",
	// "videoplayreferrersession",
	// "videoplayuniquesession",
	// "commercialsession",
	// "raidexecutesession",

	// "minutewatchedhostraidedsession",
	//"videoplayclipsreferrertimeseries",
	// "videoplayclipsreferrersession",
	//"videoplayclipscreatetimeseries",
	// "videoplayclipscreatesession",
	//
	//"chattimeseries",
	//"raidexecutetimeseries",
	"subscriptionspurchasesuccesstimeseries",
	//"videoplayuniquetimeseries",
	//"commercialtimeseries",
	//"serverfollowtimeseries",
	//"concurrentstimeseries",
}

const (
	env           = "production"
	batchInterval = 24 * 7 * time.Hour
	sleepDuration = 1 * 60 * 20 * time.Second
)

var queueURL = fmt.Sprintf("https://sqs.us-west-2.amazonaws.com/989470033077/cb-semki-%s-cron-queue", env)

func main() {
	statsdClient, err := statsd.NewClient("graphite.internal.justin.tv:8125", env, "cron", "sqs")
	if err != nil {
		log.WithError(err).Fatal("Failed to instantiate StatsD client")
		return
	}

	client := sqs.New(session.Must(session.NewSession(&aws.Config{
		Credentials: credentials.NewSharedCredentials("", "twitch-cb-aws"),
		Region:      aws.String("us-west-2"),
	})))

	batchStartTime := backfillStartTime

	for batchStartTime.Before(backfillEndTime) {
		batchEndTime := batchStartTime.Add(batchInterval)

		logEntry := log.WithFields(log.Fields{
			"start_time": batchStartTime,
			"end_time":   batchEndTime,
		})

		logEntry.Info("Starting batch")
		now := time.Now()

		for _, stat := range stats {
			input := &sqs.SendMessageInput{
				MessageAttributes: map[string]*sqs.MessageAttributeValue{
					sqswrapper.MessageAttributeKeyName: {
						DataType:    aws.String("String"),
						StringValue: aws.String(stat),
					},
					"Backfill-Start-Date": {
						DataType:    aws.String("String"),
						StringValue: aws.String(batchStartTime.Format(time.RFC3339)),
					},
					"Backfill-End-Date": {
						DataType:    aws.String("String"),
						StringValue: aws.String(batchEndTime.Format(time.RFC3339)),
					},
				},
				MessageBody: aws.String("{}"),
				QueueUrl:    aws.String(queueURL),
			}

			if err = input.Validate(); err != nil {
				logEntry.WithError(err).Fatal("Invalid input for SQS message")
				return
			}

			_, err = client.SendMessageWithContext(context.Background(), input)
			if err != nil {
				logEntry.WithError(err).Error("Failed to POST to SQS")
				continue
			}

			logEntry.WithField("stat", stat).Info("Sent to SQS")

			if err = statsdClient.Inc(fmt.Sprintf("%s.messages", stat), 1, 1); err != nil {
				logEntry.WithError(err).WithField("stat", stat).Error("Failed to increment StatsD")
			}
		}

		logEntry.WithField("duration", time.Since(now)).Info("Finished batch")
		batchStartTime = batchEndTime

		logEntry.Infof("Sleeping for %s...", sleepDuration)
		time.Sleep(sleepDuration)
	}
}
