package main

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"strconv"
	"syscall"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/experiments"
	"code.justin.tv/cb/semki/internal/clients/redshift"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"
	"code.justin.tv/cb/semki/internal/cron"

	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/semaphore"
)

func init() {
	log.Infof("Starting up cron environment %s", time.Now().UTC())
	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	env := config.Environment
	region := config.Values.AWSRegion

	semaphore := semaphore.NewWeighted(config.Values.PoolSize.Cron)

	redshiftDB, err := redshift.OpenConnection(config.Secrets.Redshift.Credentials, config.Values.Redshift.Address)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate new redshift client")
		return
	}

	ingestSQS, err := sqs.NewClient(env, region, config.Values.IngestSQSQueueURL)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate ingest sqs client")
		return
	}

	cronSQS, err := sqs.NewClient(env, region, config.Values.CronSQSQueueURL)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate cron sqs client")
		return
	}

	statsdSvcClient, err := statsd.NewClient(config.Values.Statsd.Host, config.Environment, "cron", "service")
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate svc statsd")
		return
	}

	statsdCronSQSClient, err := statsd.NewClient(config.Values.Statsd.Host, config.Environment, "cron", "sqs")
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate sqs statsd")
		return
	}

	statsdIngestSQSClient, err := statsd.NewClient(config.Values.Statsd.Host, config.Environment, "ingest", "sqs")
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate sqs statsd")
		return
	}

	experimentsClient, err := experiments.NewClient(config.Values.SpadeHost, GetEnvInt64("SPADE_BACKLOG_SIZE", 1024))
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate experiments")
		return
	}

	server := &http.Server{
		Addr: ":8000",
		Handler: cron.NewServer(&cron.ServerParams{
			CronSQS:            cronSQS,
			IngestSQS:          ingestSQS,
			Redshift:           redshiftDB,
			CronSvcStatsd:      statsdSvcClient,
			CronSQSStatsd:      statsdCronSQSClient,
			Experiments:        experimentsClient,
			IngestSQSStatsd:    statsdIngestSQSClient,
			Pool:               semaphore,
			BackfillInProgress: (os.Getenv("BACKFILL_IN_PROGRESS") == "true"),
		}),
	}

	go func() {
		log.Info("cron: server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("cron: server failed fatally while listening")
		}
	}()

	graceful(server)
}

const timeout = 5 * time.Second

func graceful(server *http.Server) {
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, os.Interrupt, syscall.SIGTERM)

	<-stop

	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()

	log.Infof("cron: shutting down server with %s timeout", timeout)

	if err := server.Shutdown(ctx); err != nil {
		log.WithError(err).Fatal("cron: server failed to shut down")
	} else {
		log.Info("cron: gracefully shut down server")
	}
}

// GetEnvInt64 gets an environment variable or returns a default
func GetEnvInt64(key string, dft int64) int64 {
	value := os.Getenv(key)
	if value == "" {
		return dft
	}

	intValue, err := strconv.ParseInt(value, 10, 64)
	if err != nil {
		message := fmt.Sprintf("Failed to parse env: %s=%s as int64", key, value)
		panic(message)
	}

	return intValue
}
