package main

import (
	"context"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"code.justin.tv/cb/semki/config"
	"code.justin.tv/cb/semki/internal/clients/achievements"
	"code.justin.tv/cb/semki/internal/clients/clips"
	"code.justin.tv/cb/semki/internal/clients/dynamo"
	"code.justin.tv/cb/semki/internal/clients/pushy"
	"code.justin.tv/cb/semki/internal/clients/sqs"
	"code.justin.tv/cb/semki/internal/clients/statsd"
	"code.justin.tv/cb/semki/internal/ingest"
	"code.justin.tv/foundation/twitchclient"
	users "code.justin.tv/web/users-service/client/usersclient_internal"

	log "github.com/sirupsen/logrus"
)

func init() {
	log.Infof("Starting up ingest environment %s", time.Now().UTC())
	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	env := config.Environment
	region := config.Values.AWSRegion

	achievementsClient, err := achievements.NewClient(config.Values.AchievementsHost)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate achievements")
		return
	}

	clipsClient, err := clips.NewClient(config.Values.ClipsHost)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate clips")
		return
	}

	dynamoClient, err := dynamo.NewClient(env, region)
	if err != nil {
		log.WithError(err).Fatal("ingest: failed to instantiate new dynamodb client")
		return
	}

	statsdSvcClient, err := statsd.NewClient(config.Values.Statsd.Host, config.Environment, "ingest", "service")
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate svc statsd")
		return
	}

	statsdSQSClient, err := statsd.NewClient(config.Values.Statsd.Host, config.Environment, "ingest", "sqs")
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate sqs statsd")
		return
	}

	pushyClient, err := pushy.NewClient(env, region, config.Values.PushySNSARN, statsdSvcClient)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate pushy")
		return
	}

	ingestSQS, err := sqs.NewClient(env, region, config.Values.IngestSQSQueueURL)
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate ingest sqs client")
		return
	}

	usersClient, err := users.NewClient(twitchclient.ClientConf{
		Host:           config.Values.UsersURL,
		Stats:          statsdSQSClient,
		TimingXactName: "users_service",
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 10,
		},
	})
	if err != nil {
		log.WithError(err).Fatal("cron: failed to instantiate ingest users client")
		return
	}

	server := &http.Server{
		Addr: ":8000",
		Handler: ingest.NewServer(&ingest.ServerParams{
			IngestSQS:       ingestSQS,
			Achievements:    achievementsClient,
			Clips:           clipsClient,
			DynamoDB:        dynamoClient,
			Pushy:           pushyClient,
			IngestSvcStatsd: statsdSvcClient,
			IngestSQSStatsd: statsdSQSClient,
			Users:           usersClient,
		}),
	}

	go func() {
		log.Info("ingest: server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("ingest: server failed fatally while listening")
		}
	}()
	graceful(server)
}

const timeout = 5 * time.Second

func graceful(server *http.Server) {
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, os.Interrupt, syscall.SIGTERM)

	<-stop

	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()

	log.Infof("shutting down server with %s timeout", timeout)

	if err := server.Shutdown(ctx); err != nil {
		log.WithError(err).Fatal("server failed to shut down")
	} else {
		log.Info("gracefully shut down server")
	}
}
