package achievements

import (
	"context"

	"code.justin.tv/cb/achievements/client/achievementsinternal"
	"code.justin.tv/cb/achievements/view"
	"code.justin.tv/chat/pushy/client/events"
	"code.justin.tv/foundation/twitchclient"
)

// Achievements gets achievements data for a channel
type Achievements interface {
	GetQuestData(ctx context.Context, channelID string) (events.QuestData, error)
}

// Client is a wrapper for an achievementsinternal.Client client
type Client struct {
	achievements achievementsinternal.Client
}

// NewClient creates an instance of an Achievements Client
func NewClient(achievementsHost string) (Achievements, error) {
	achievements, err := achievementsinternal.NewClient(twitchclient.ClientConf{
		Host: achievementsHost,
	})
	if err != nil {
		return nil, err
	}

	return &Client{
		achievements: achievements,
	}, nil
}

// GetQuestData gathers in-progress quest data for a channel in Pushy's quest type.
// An incomplete "Path to Affiliate" quest is prioritized over an incomplete "Path to Partner" quest.
// If both quests are completed, then the returned quest data is empty.
func (c *Client) GetQuestData(ctx context.Context, channelID string) (events.QuestData, error) {
	rewardsIncluded := false
	quests, err := c.achievements.GetV1Quests(ctx, channelID, view.V1QuestsRequestParams{
		RewardsIncluded: &rewardsIncluded,
	}, nil)
	if err != nil {
		return events.QuestData{}, err
	}

	var lowestIncompleteQuest *view.V1Quest

	for _, quest := range quests.Data.Quests {
		if quest.IsComplete {
			continue
		}

		switch quest.Key {
		case "path_to_affiliate":
			lowestIncompleteQuest = quest
		case "path_to_partner":
			if lowestIncompleteQuest == nil {
				lowestIncompleteQuest = quest
			}
		}
	}

	if lowestIncompleteQuest == nil {
		return events.QuestData{}, nil
	}

	return extractQuestData(lowestIncompleteQuest), nil
}

func extractQuestData(quest *view.V1Quest) events.QuestData {
	questData := events.QuestData{
		Key: quest.Key,
	}

	for _, progress := range quest.Progress {
		if progress != nil && progress.Metric == "unique_days" {
			questData.Days = progress.Progress
			questData.DaysCap = progress.ProgressCap
		}
		if progress != nil && progress.Metric == "follow_count" {
			questData.Followers = progress.Progress
			questData.FollowersCap = progress.ProgressCap
		}
		if progress != nil && progress.Metric == "hour_count" {
			questData.Hours = progress.Progress
			questData.HoursCap = progress.ProgressCap
		}
		if progress != nil && progress.Metric == "average_ccu" {
			questData.Viewers = progress.Progress
			questData.ViewersCap = progress.ProgressCap
		}
	}

	return questData
}
