package dynamo

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/cb/semki/internal/awscredentials"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/pkg/errors"
)

// Client is a wrapper for the dynamodb api
type Client struct {
	env      string
	dynamoDB dynamodbiface.DynamoDBAPI
}

// NewClient creates an instance of an sns client
func NewClient(env string, region string) (*Client, error) {
	conf := &aws.Config{
		Credentials: awscredentials.New(env, region),
		Region:      aws.String(region),
	}

	sess, err := session.NewSession(conf)
	if err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to initialize dynamo db client")
	}

	return &Client{
		env:      env,
		dynamoDB: dynamodb.New(sess),
	}, nil
}

// BatchPut gets a list of write reqs and batch puts them into dynamo
func (c *Client) BatchPut(ctx context.Context, puts []*dynamodb.WriteRequest, table string, backoff int) error {
	time.Sleep(time.Duration(backoff) * time.Second)
	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			table: puts,
		},
	}

	msg := fmt.Sprintf("dynamodb: failed to batch put into table %s", table)
	out, err := c.dynamoDB.BatchWriteItemWithContext(ctx, input)
	if err != nil {
		return errors.Wrap(err, msg)
	}

	if len(out.UnprocessedItems) > 0 {
		return errors.New(msg)
	}

	return nil
}

// BatchGet performs a batch get request on dynamo
func (c *Client) BatchGet(ctx context.Context, input *dynamodb.BatchGetItemInput) (*dynamodb.BatchGetItemOutput, error) {
	return c.dynamoDB.BatchGetItemWithContext(ctx, input)
}

// Query performs a batch get request on dynamo
func (c *Client) Query(ctx context.Context, input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	return c.dynamoDB.QueryWithContext(ctx, input)
}
