package dynamo

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/service/dynamodb"
)

const (
	//DynamoTimeFormat is how we format timestamps in dynamo
	DynamoTimeFormat = "2006-01-02 15:04:05"

	// BatchSize are the amount of rows we will put in a batch put into dynamo
	BatchSize = 20
)

//Dynamo is the interface for interacting with achievement dynamo tables
type Dynamo interface {
	// BatchPut takes a list of put write requests and sends them to dynamo
	BatchPut(ctx context.Context, puts []*dynamodb.WriteRequest, table string, backoff int) error
	// BatchGet performs a batch get request on dynamo
	BatchGet(ctx context.Context, input *dynamodb.BatchGetItemInput) (*dynamodb.BatchGetItemOutput, error)
	// Query performs a batch get request on dynamo
	Query(ctx context.Context, input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error)
}

// Timestamp our tables store time in "2006-01-02 15:04:05" format
// this is a custom unmarshaller that will parse the times
// appropriately
type Timestamp struct {
	Converted time.Time
}

// MarshalDynamoDBAttributeValue is the custom marshaller for the Timestamp type
func (propTime *Timestamp) MarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	timeObj := propTime.Converted
	timeString := timeObj.Format(DynamoTimeFormat)
	av.S = &timeString

	return nil
}

// UnmarshalDynamoDBAttributeValue is the custom unmarshaller for the Timestamp type
func (propTime *Timestamp) UnmarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	timeString := av.S
	if timeString == nil {
		return nil
	}

	timeObj, err := time.Parse(DynamoTimeFormat, *timeString)
	if err != nil {
		return err
	}

	propTime.Converted = timeObj
	return nil
}
