package experiments

import (
	"context"
	"fmt"

	exp "code.justin.tv/discovery/experiments"

	log "github.com/sirupsen/logrus"
)

const (
	// Stream Summary experiment details
	streamSummaryExperimentName = "stream_summary_email"
	streamSummaryExperimentID   = "781530f1-5a77-449c-9d1e-604db8050127"
	streamSummaryControlGroup   = "control"
	streamSummaryTestGroup      = "test"
)

// Experiment gets information regarding a minixperiment experiment
type Experiment interface {
	IsInStreamSummaryExperiment(ctx context.Context, userID string) (bool, error)
}

// Client is a wrapper for an experiments.Experiments client
type Client struct {
	experiments exp.Experiments
}

// NewClient creates an instance of an Experiment Client
func NewClient(spadeHost string, spadeBacklogSize int64) (*Client, error) {
	config := exp.Config{
		Platform: exp.TwilightPlatform,
		ErrorHandler: func(err error) {
			errMsg := fmt.Sprintf("experiments: %v", err)
			log.WithError(err).Error(errMsg)
		},
	}

	if spadeBacklogSize == 0 {
		config.DisableSpadeTracking = true
	} else {
		config.SpadeEndpoint = spadeHost
		config.SpadeBacklogSize = spadeBacklogSize
	}

	exps, err := exp.New(config)
	if err != nil {
		return nil, err
	}

	client := &Client{
		experiments: exps,
	}

	err = client.setUpDefaultsAndOverrides()
	if err != nil {
		return nil, err
	}

	return client, nil
}

// IsInStreamSummaryExperiment returns if a user is in the stream summary experiment's test group
func (c *Client) IsInStreamSummaryExperiment(ctx context.Context, userID string) (bool, error) {
	group, err := c.experiments.Treat(
		exp.UserType,
		streamSummaryExperimentName,
		streamSummaryExperimentID,
		userID,
	)
	if err != nil {
		return false, err
	}

	return (group == streamSummaryTestGroup), nil
}

func (c *Client) setUpDefaultsAndOverrides() error {
	// Set up stream summary experiment
	c.experiments.RegisterDefault(
		exp.UserType,
		streamSummaryExperimentName,
		streamSummaryExperimentID,
		streamSummaryControlGroup,
	)
	err := c.experiments.RegisterOverride(
		exp.UserType, streamSummaryExperimentName, streamSummaryExperimentID, streamSummaryTestGroup,
		"19629",     // bill
		"36002799",  // dathass
		"104354422", // danly
		"138357543", // signal
		"141594393", // phunktional
		"153817938", // forever_rhapsody
		"160678210", // littlergromit
		"188973877", // sprezzatura8
		"136770106",
		"12745158",
		"62975170",
		"473396",
		"157698494",
		"19571752",
		"107591830",
		"35790122",
		"107805111",
		"100409814",
		"36584347",
		"221746722",
		"67054208",
		"49297948",
		"55347167",
		"136288163",
		"170649963",
		"28318780",
		"26025393",
		"57674864",
		"28337972",
		"180833069",
		"47007372",
		"137256300",
		"1839275",
		"76454775",
		"225960490",
		"69932354",
		"23406143",
		"125081985",
		"106064402",
		"32374525",
		"257788195",
		"19312448",
		"67771987",
		"43893913",
		"65158718",
		"16151006",
		"22602760",
		"161128364",
		"40070491",
		"62564659",
		"26498204",
		"50707402",
		"7990962",
		"137317789",
		"91003895",
		"82114918",
		"182723142",
		"63375090",
		"21612261",
		"45867146",
		"6353604",
		"87210460",
		"120424257",
		"104354422",
		"1315120",
		"19079059",
		"186243532",
		"32454949",
		"107167633",
		"57623709",
		"38426833",
		"471453",
		"215491810",
		"158238984",
		"45776283",
		"58104889",
		"40863935",
		"14546066",
		"39361778",
		"63759512",
		"23429329",
		"269643254",
		"55492306",
		"22129289",
		"76846095",
		"160244853",
		"46767338",
		"138431115",
		"14204917",
		"46024993",
		"29726548",
		"138841370",
		"150573462",
		"30515034",
		"151051203",
		"20814295",
		"68301096",
		"49307088",
		"28213853",
		"135093069",
		"255942031",
		"158038007",
		"246831209",
		"143526973",
		"15004026",
		"85901348",
		"118762610",
		"38704858",
		"92672952",
		"13243026",
		"40022383",
		"111740869",
		"32274396",
		"94011",
		"54591721",
		"228722817",
		"32316661",
		"65203692",
		"36893878",
		"108707191",
		"30015604",
		"47969748",
		"159216544",
		"49370009",
		"73430381",
		"139157487",
		"158832315",
		"29432474",
		"21588923",
		"134155019",
		"13405587",
		"115090392",
		"40529315",
		"138289128",
		"54703138",
		"95540308",
		"51837161",
		"29214101",
		"92714955",
		"22158994",
		"42832269",
		"14556978",
		"182891647",
		"237726346",
		"73906908",
		"78857480",
		"237755203",
		"379382",
		"50208846",
		"51756334",
		"123973688",
		"27409150",
		"157853062",
		"39173706",
		"76867",
		"43445044",
		"40514767",
		"113408920",
		"102439686",
		"42197189",
		"167136666",
		"74681932",
		"69183659",
		"46056451",
		"178109690",
		"27795595",
		"16689163", // Criken
	)
	if err != nil {
		return err
	}

	return nil
}
