package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
)

const (
	clipsLock = `
    LOCK analysis.clips
    `

	tempClips = `
    CREATE TEMP TABLE new_clip_views AS (
    	SELECT *,
        	count(*) AS view_count
        FROM (
	      	SELECT vod_id AS clip_id,
	        date_trunc('day', time_utc) AS view_day_utc,
            CASE WHEN referrer_url ILIKE '%www.twitch.tv%' THEN 'twitch'
	             WHEN referrer_url ILIKE '%my-clip%' THEN 'twitch'
                 WHEN referrer_url ILIKE '%clips.twitch.tv%' THEN 'twitch'
	             WHEN referrer_url ILIKE '%.reddit.%' THEN 'reddit'
	             WHEN referrer_url ILIKE '%.redd.it%' THEN 'reddit'
                 WHEN referrer_url ILIKE '%reddit%' THEN 'reddit'
	             WHEN referrer_url ILIKE '%embedly%' THEN 'reddit'
	             WHEN referrer_url ILIKE '%facebook.%' THEN 'facebook'
	             WHEN referrer_url ILIKE '%http://t.co%' THEN 'twitter'
                 WHEN referrer_url ILIKE '%https://t.co%' THEN 'twitter'
	             WHEN referrer_url ILIKE '%twitter%' THEN 'twitter'
	        ELSE 'other'
	        END AS  referrer
	    	FROM 	tahoe_recent."video-play"
			WHERE   date >= $1
	        AND     date < $2
	        AND 	vod_type = 'clip'
	        AND 	vod_id IS NOT NULL
	        AND 	vod_id SIMILAR TO '[0-9]%'
      	)
        WHERE 	view_day_utc >= $3
        AND 	view_day_utc < $4
		GROUP BY clip_id, view_day_utc, referrer
    )
    `

	clipsPurge = `
    DELETE  FROM analysis.clips
    USING 	new_clip_views
    WHERE   clips.clip_id = new_clip_views.clip_id
    AND     clips.view_date_utc = new_clip_views.view_day_utc
    AND 	clips.referrer = new_clip_views.referrer
    `

	clipsInsert = `
    INSERT INTO analysis.clips(clip_id, referrer, view_date_utc, view_count) (
    	SELECT clip_id, referrer, view_day_utc, view_count from new_clip_views
    )
    `

	clipsCleanup = `
    DROP TABLE IF EXISTS new_clip_views
    `
)

// CalculateClips builds referral maps for clips from video-play events. The referral maps
// are stored by utc days of video-play events
func (c *Client) CalculateClips(ctx context.Context, start time.Time, end time.Time) error {
	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	txn, err := c.BeginTx(ctx, nil)
	if err != nil {
		return errors.Wrap(err, "redshift calculate clips: failed to start tranasction")
	}

	defer func() {
		if err != nil {
			err = txn.Rollback()

		}
		err = txn.Commit()
	}()

	// lock the table while we are performing clips recalcs
	_, err = txn.ExecContext(ctx, clipsLock)
	if err != nil {
		return err
	}

	// build a temp table of all clip views generated from these video plays
	_, err = txn.ExecContext(ctx, tempClips, aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return err
	}

	// remove existing entries for this clip view day
	_, err = txn.ExecContext(ctx, clipsPurge)
	if err != nil {
		return err
	}

	// copy the new values over
	_, err = txn.ExecContext(ctx, clipsInsert)
	if err != nil {
		return err
	}

	// cleanup the temp table
	_, err = txn.ExecContext(ctx, clipsCleanup)
	if err != nil {
		return err
	}

	return nil
}
