package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetBroadcastIDSessionAggregates queries Redshift to retrieves broadcast ids (VOD division of streams) per session
func (c *Client) GetBroadcastIDSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]BroadcastIDAggregate, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_minute_broadcast AS
        (    SELECT  channel_id,
                     broadcast_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.minute_broadcast
             WHERE   date >= $3
             AND     date < $4
        )

        SELECT a.channel_id,
               a.segment_start_time,
               listagg(distinct b.broadcast_id, ',') AS broadcast_ids
        FROM recent_sessions a
        INNER JOIN recent_minute_broadcast b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []BroadcastIDAggregate
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get broadcast_ids: failed to query broadcast ids by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get broadcast_ids: failed to close rows when querying broadcast ids by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var broadcast BroadcastIDAggregate
		err = rows.Scan(
			&broadcast.ChannelID,
			&broadcast.SegmentStartTime,
			&broadcast.BroadcastIDs,
		)

		results = append(results, broadcast)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get broadcast_ids: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
