package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetChatSessionAggregates queries Redshift to retrieves chat aggregates
// for the server_chat_message spade event
func (c *Client) GetChatSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]ChatAggregates, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >=$1
             AND     segment_start_time < $2
        ),   recent_chat_messages AS
        (    SELECT  channel_id,
                     user_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.server_chat_message
             WHERE   date >= $3
             AND     date < $4
        )
        SELECT a.channel_id,
               a.segment_start_time,
               count(DISTINCT b.user_id) AS unique_chatters,
               count(b.channel_id) AS total_messages
        FROM recent_sessions a
        INNER JOIN recent_chat_messages b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ChatAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get server_chat_message session: failed to query chat messages by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get server_chat_message session: failed to close rows when querying chat messages by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg ChatAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.UniqueChatters,
			&agg.TotalMessages,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get server_chat_message session: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
