package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetChatTimeseries queries Redshift to retrieves server_chat_message aggregates
// in 5 minute chunks
func (c *Client) GetChatTimeseries(ctx context.Context, start time.Time, end time.Time) ([]ChatTimeseriesUnit, error) {
	statement := `
		WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time <= $2
             AND 	 segment_end_time >= $1
        ), recent_chat_messages AS
        (    SELECT  channel_id,
                     user_id,
                     time_utc,
                     date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS grouping
             FROM    tahoe_recent.server_chat_message
             WHERE   date >= $3
             AND     date < $4
             AND     channel_id IS NOT NULL
        )

        SELECT a.channel_id,
               b.grouping AS timestamp,
               count(DISTINCT b.user_id) AS unique_chatters,
               count(b.channel_id) AS total_messages
        FROM recent_sessions a
        INNER JOIN recent_chat_messages b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        WHERE b.grouping >= $1
        	AND b.grouping < $2
        GROUP BY a.channel_id, b.grouping
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ChatTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get server_chat_message timeseries: failed to query chat messages by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get server_chat_message timeseries: failed to close rows when querying chat messages by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit ChatTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.UniqueChatters,
			&unit.TotalMessages,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get server_chat_message timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
