package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetCommercialLengthSessionAggregates queries Redshift to retrieves commercial session aggregates
// for the commercial spade event
func (c *Client) GetCommercialLengthSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]CommercialAggregates, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_commercial_length AS
        (    SELECT  channel_id,
                     length,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.commercial
             WHERE   date >= $3
             AND     date < $4
        )
        SELECT a.channel_id,
               a.segment_start_time,
               SUM(b.length) AS commercial_length,
               COUNT(b.length) AS commercial_count
        FROM recent_sessions a
        INNER JOIN recent_commercial_length b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []CommercialAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get commercial session: failed to query commercial length by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get commercial session: failed to close rows when querying commercial length by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg CommercialAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.CommercialLength,
			&agg.CommercialCount,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get commercial: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
