package redshift

import (
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetConcurrentsSessionAggregatesEndingAt queries Redshift to retrieves session aggregates for
// the channel_concurrents spade event
func (c *Client) GetConcurrentsSessionAggregatesEndingAt(ctx context.Context, start time.Time, end time.Time) ([]ConcurrentAggregates, error) {
	return c.getConcurrentsSessionAggregatesWithTimeKey(ctx, start, end, sessionsEndingAt)
}

// GetConcurrentsSessionAggregates queries Redshift to retrieves session aggregates for
// the channel_concurrents spade event
func (c *Client) GetConcurrentsSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]ConcurrentAggregates, error) {
	return c.getConcurrentsSessionAggregatesWithTimeKey(ctx, start, end, sessionsStartingAt)
}

// GetConcurrentsSessionAggregates queries Redshift to retrieves session aggregates for
// the channel_concurrents spade event
func (c *Client) getConcurrentsSessionAggregatesWithTimeKey(ctx context.Context, start time.Time, end time.Time, key string) ([]ConcurrentAggregates, error) {
	statement := fmt.Sprintf(`
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   %v >= $1
             AND     %v < $2
        ),   recent_channel_concurrents AS
        (    SELECT  channel_id,
                     total,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.channel_concurrents
             WHERE   date >= $3
             AND     date < $4
        ),  ranked_concurrents AS
        (
            SELECT a.channel_id,
                a.segment_start_time,
                b.total AS ccu,
                first_value(b.time_utc) OVER (PARTITION BY a.channel_id, a.segment_start_time ORDER BY b.total DESC, b.time_utc DESC ROWS BETWEEN UNBOUNDED PRECEDING AND UNBOUNDED FOLLOWING) AS peak_time
            FROM recent_sessions a
            INNER JOIN recent_channel_concurrents b
            ON a.segment_start_time <= b.time_utc
                AND a.segment_end_time >= b.time_utc
                AND a.channel_id = b.channel_id
        )

        SELECT channel_id,
            segment_start_time,
            avg(ccu::float) AS average_ccu,
            sum(ccu) AS total_minutes_watched,
            max(ccu) AS max_ccu,
            max(peak_time) as peak_time
        FROM ranked_concurrents
        GROUP BY channel_id, segment_start_time
        ORDER BY channel_id DESC, segment_start_time DESC
    `, key, key)

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ConcurrentAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get channel_concurrents aggregates: failed to query average ccu by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get channel_concurrents aggregates: failed to close rows when querying average ccu by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg ConcurrentAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.AverageConcurrents,
			&agg.TotalMinutesWatched,
			&agg.MaxConcurrents,
			&agg.PeakCCUTime,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get channel_concurrents aggregates: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
