package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetConcurrentsTimeseries queries Redshift to retrieves channel_concurrents aggregates
// in 5 minute chunks
func (c *Client) GetConcurrentsTimeseries(ctx context.Context, start time.Time, end time.Time) ([]ConcurrentTimeseriesUnit, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time <= $2
             AND 	 segment_end_time >= $1
        ), recent_channel_concurrents AS
        (    SELECT  channel_id,
                     total,
                     date_trunc('minute', time_utc) AS time_utc,
                     date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS grouping
             FROM    tahoe_recent.channel_concurrents
             WHERE   date >= $3
             AND     date < $4
             AND     channel_id IS NOT NULL
        )

        SELECT a.channel_id,
               b.grouping AS timestamp,
               avg(b.total::float) AS average_ccu
        FROM recent_sessions a
        INNER JOIN recent_channel_concurrents b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        WHERE b.grouping >= $1
        	AND b.grouping < $2
        GROUP BY a.channel_id, b.grouping
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ConcurrentTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get channel_concurrents timeseries: failed to query average ccu by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get channel_concurrents timeseries: failed to close rows when querying average ccu by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit ConcurrentTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.AverageConcurrents,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get channel_concurrents timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
