package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetMinuteWatchedHostRaidAggregates queries Redshift to retrieves session aggregates for
// of hostedMW raid sourced MW and Total MW (to calculate host/raid %)
func (c *Client) GetMinuteWatchedHostRaidAggregates(ctx context.Context, start time.Time, end time.Time) ([]MinuteWatchedHostRaidAggregates, error) {
	statement := `
        WITH recent_sessions AS
        (
            SELECT  *
            FROM    analysis.sessions
            WHERE   segment_start_time >= $1
            AND     segment_start_time < $2
        ),

        recent_minute_watched AS
        (
            SELECT   channel_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     SUM(CASE WHEN (content = 'raid_channel' AND (host_channel = '' OR host_channel IS NULL)) THEN 1 ELSE 0 END) AS raided_mw,
                     SUM(CASE WHEN (host_channel <> '' AND host_channel IS NOT NULL) THEN 1 ELSE 0 END) AS hosted_mw,
                     COUNT(*) AS total_mw
             FROM    tahoe_recent."minute-watched"
             WHERE   date >= $3
             AND     date < $4
             AND 	 (live IS NULL OR live IS true)
             GROUP BY channel_id, date_trunc('minute', time_utc)
        )

        SELECT a.channel_id,
               a.segment_start_time,
               SUM(b.raided_mw) as raided_mw,
               SUM(b.hosted_mw) as hosted_mw,
               SUM(b.total_mw) as total_mw
        FROM   recent_sessions a
        INNER JOIN recent_minute_watched b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []MinuteWatchedHostRaidAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get minute_watched host aggregates: failed to query")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get minute_watched host aggregates: failed to close rows when querying"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg MinuteWatchedHostRaidAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.Raided,
			&agg.Hosted,
			&agg.Total,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get minute_watched host aggregates: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
