package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetRaidExecuteSessionAggregates queries Redshift to retrieves raid_execute aggregates
// for the raid_execute spade event
func (c *Client) GetRaidExecuteSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]RaidExecuteAggregates, error) {
	statement := `
		WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ),   recent_raid_executes AS
        (    SELECT  viewer_count,
                     raid_target_id as target_id,
                     raid_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.raid_execute
             WHERE   date >= $3
             AND     date < $4
        )
        SELECT a.channel_id,
               a.segment_start_time,
               COUNT(raid_id) as number_of_raids,
               SUM(viewer_count) as viewer_count
        FROM recent_sessions a
        INNER JOIN recent_raid_executes b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.target_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []RaidExecuteAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get raid_execute session: failed to query raid stats by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get raid_execute session: failed to close rows when querying raid stats by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg RaidExecuteAggregates
		err = rows.Scan(
			&agg.TargetID,
			&agg.SegmentStartTime,
			&agg.NumberOfRaids,
			&agg.ViewersCount,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get raid_execute session: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
