package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetRaidExecuteTimeseries queries Redshift to retrieves raid_execute aggregates
// in 5 minute chunks
func (c *Client) GetRaidExecuteTimeseries(ctx context.Context, start time.Time, end time.Time) ([]RaidExecuteTimeseriesUnit, error) {
	statement := `
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time <= $2
             AND     segment_end_time >= $1
        ), recent_raid_executes AS
        (    SELECT  viewer_count,
                     raid_target_id as target_id,
                     raid_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS grouping
             FROM    tahoe_recent.raid_execute
             WHERE   date >= $3
             AND     date < $4
        )

        SELECT a.channel_id,
               b.grouping AS timestamp,
               COUNT(raid_id) as number_of_raids,
               SUM(viewer_count) as viewer_count
        FROM recent_sessions a
        INNER JOIN recent_raid_executes b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.target_id
        WHERE b.grouping >= $1
            AND b.grouping < $2
        GROUP BY a.channel_id, b.grouping
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []RaidExecuteTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get raid_execute timeseries: failed to query raid executions by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get raid_execute timeseries: failed to close rows when querying raid executions by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit RaidExecuteTimeseriesUnit
		err = rows.Scan(
			&unit.TargetID,
			&unit.Timestamp,
			&unit.NumberOfRaids,
			&unit.ViewersCount,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get raid_execute timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
