package redshift

import (
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetRecentSessionsEndingAt queries Redshift to retrieves sessions ending in the requested time range
func (c *Client) GetRecentSessionsEndingAt(ctx context.Context, start time.Time, end time.Time) ([]Session, error) {
	return c.getRecentSessionsWithTimeKey(ctx, start, end, sessionsEndingAt)
}

// GetRecentSessions queries Redshift to retrieves sessions starting in the requested time range
func (c *Client) GetRecentSessions(ctx context.Context, start time.Time, end time.Time) ([]Session, error) {
	return c.getRecentSessionsWithTimeKey(ctx, start, end, sessionsStartingAt)
}

// GetRecentSessionsWithTimeKey queries Redshift to retrieves sessions starting or ending in the requested time range
func (c *Client) getRecentSessionsWithTimeKey(ctx context.Context, start time.Time, end time.Time, key string) ([]Session, error) {
	statement := fmt.Sprintf(`
        SELECT  is_segmented,
            segment_start_time,
            segment_end_time,
            channel_id
        FROM    analysis.sessions
        WHERE   %v >= $1
        AND     %v < $2
        ORDER BY channel_id ASC;
    `, key, key)

	var results []Session
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get channel_concurrents timeseries: failed to query average ccu by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get channel_concurrents timeseries: failed to close rows when querying average ccu by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var session Session
		err = rows.Scan(
			&session.IsSegmented,
			&session.SegmentStartTime,
			&session.SegmentEndTime,
			&session.ChannelID,
		)

		results = append(results, session)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift calculate session: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
