package redshift

import (
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetNewFollowersSessionAggregatesEndingAt queries Redshift to retrieves server_follow session aggregates
// for the server_follow spade event
func (c *Client) GetNewFollowersSessionAggregatesEndingAt(ctx context.Context, start time.Time, end time.Time) ([]ServerFollowAggregates, error) {
	return c.getNewFollowersSessionAggregatesWithTimeKey(ctx, start, end, sessionsEndingAt)
}

// GetNewFollowersSessionAggregates queries Redshift to retrieves server_follow session aggregates
// for the server_follow spade event
func (c *Client) GetNewFollowersSessionAggregates(ctx context.Context, start time.Time, end time.Time) ([]ServerFollowAggregates, error) {
	return c.getNewFollowersSessionAggregatesWithTimeKey(ctx, start, end, sessionsStartingAt)
}

// GetNewFollowersSessionAggregates queries Redshift to retrieves server_follow session aggregates
// for the server_follow spade event
func (c *Client) getNewFollowersSessionAggregatesWithTimeKey(ctx context.Context, start time.Time, end time.Time, key string) ([]ServerFollowAggregates, error) {
	statement := fmt.Sprintf(`
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   %v >= $1
             AND     %v < $2
        ),   recent_new_followers AS
        (    SELECT  target_user_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.server_follow
             WHERE   date >= $3
             AND     date < $4
        )

        SELECT a.channel_id,
               a.segment_start_time,
               count(b.target_user_id) AS new_followers
        FROM recent_sessions a
        INNER JOIN recent_new_followers b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.target_user_id
        GROUP BY a.channel_id, a.segment_start_time
        ORDER BY a.channel_id DESC, a.segment_start_time DESC
    `, key, key)

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ServerFollowAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get server_follow session: failed to query new followers by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get server_follow session: failed to close rows when querying average new followers by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg ServerFollowAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.NewFollowers,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get server_follow: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
