package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetNewFollowersTimeseries queries Redshift to retrieves server_follow aggregates
// in 5 minute chunks
func (c *Client) GetNewFollowersTimeseries(ctx context.Context, start time.Time, end time.Time) ([]ServerFollowTimeseriesUnit, error) {
	statement := `
		WITH recent_new_followers AS
        (    SELECT  target_user_id,
                     date_trunc('minute', time_utc) AS time_utc,
                     date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS grouping
             FROM    tahoe_recent.server_follow
             WHERE   date >= $1
             AND     date < $2
             AND     target_user_id IS NOT NULL
        )

        SELECT target_user_id as channel_id,
               grouping AS timestamp,
               count(*) AS new_followers
        FROM recent_new_followers
        WHERE grouping >= $3
        	AND grouping < $4
        GROUP BY target_user_id, grouping
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []ServerFollowTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get server_follow timeseries: failed to query newfollowers by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get server_follow timeseries: failed to close rows when querying newfollowers by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit ServerFollowTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.NewFollowers,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get server_follow timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
