package redshift

import (
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetSubscriptionPurchaseSuccessAggregatesEndingAt queries Redshift to retrieves subscription success purchase events aggregates
// for the subscription_purchase_success spade event
func (c *Client) GetSubscriptionPurchaseSuccessAggregatesEndingAt(ctx context.Context, start time.Time, end time.Time) ([]SubscriptionPurchaseSuccessAggregates, error) {
	return c.getSubscriptionPurchaseSuccessAggregatesWithTimeKey(ctx, start, end, sessionsEndingAt)
}

// GetSubscriptionPurchaseSuccessAggregates queries Redshift to retrieves subscription success purchase events aggregates
// for the subscription_purchase_success spade event
func (c *Client) GetSubscriptionPurchaseSuccessAggregates(ctx context.Context, start time.Time, end time.Time) ([]SubscriptionPurchaseSuccessAggregates, error) {
	return c.getSubscriptionPurchaseSuccessAggregatesWithTimeKey(ctx, start, end, sessionsStartingAt)
}

// GetSubscriptionPurchaseSuccessAggregates queries Redshift to retrieves subscription success purchase events aggregates
// for the subscription_purchase_success spade event
func (c *Client) getSubscriptionPurchaseSuccessAggregatesWithTimeKey(ctx context.Context, start time.Time, end time.Time, key string) ([]SubscriptionPurchaseSuccessAggregates, error) {
	statement := fmt.Sprintf(`
        WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   %v >= $1
             AND     %v < $2
        ),   converted_timestamps AS
		(    SELECT  channel_id,
					 quantity,
                     convert_timezone('PST8PDT', 'GMT', client_time) AS timestamp_utc
             FROM    tahoe_recent.subscription_purchase_success
             WHERE   date >= $3
             AND     date < $4
             AND     channel_id IS NOT NULL
        ),   recent_new_subs AS
		(    SELECT  channel_id,
					 quantity,
                     date_trunc('minute', ct.timestamp_utc) AS timestamp_utc_converted
             FROM    converted_timestamps AS ct
        )

        SELECT a.channel_id,
               a.segment_start_time,
               sum(b.quantity) AS new_subs
        FROM recent_sessions a
        INNER JOIN recent_new_subs b
        ON a.segment_start_time <= b.timestamp_utc_converted
            AND a.segment_end_time >= b.timestamp_utc_converted
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
        ORDER BY a.channel_id DESC, a.segment_start_time DESC
    `, key, key)

	startDate := start.Truncate(day).AddDate(0, 0, -1).Format(SQLTimeFormat)
	endDate := end.Truncate(day).AddDate(0, 0, 2).Format(SQLTimeFormat)
	clientStartTime := start.Format(SQLTimeFormat)
	clientEndTime := end.Format(SQLTimeFormat)

	var results []SubscriptionPurchaseSuccessAggregates
	rows, err := c.QueryContext(ctx, statement, clientStartTime, clientEndTime, startDate, endDate)
	if err != nil {
		return nil, errors.Wrap(err, "redshift get subscription_purchase_success aggregates: failed to query new subscriptions count by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get subscription_purchase_success session: failed to close rows when querying new subscriptions count by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg SubscriptionPurchaseSuccessAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.NewSubs,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get subscription_purchase_success session: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
