package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetSubscriptionPurchaseSuccessTimeSeries queries Redshift to retrieves subscription_purchase_success aggregates
// in 5 minute chunks
func (c *Client) GetSubscriptionPurchaseSuccessTimeSeries(ctx context.Context, start time.Time, end time.Time) ([]SubscriptionPurchaseSuccessTimeseriesUnit, error) {
	statement := `
		WITH converted_timestamps AS
		(    SELECT  channel_id,
					 quantity,
                     convert_timezone('PST8PDT', 'GMT', client_time) AS timestamp_utc
             FROM    tahoe_recent.subscription_purchase_success
             WHERE   date >= $1
             AND     date < $2
             AND     channel_id IS NOT NULL
        ),
        recent_new_subs AS
		(    SELECT  channel_id,
					 quantity,
                     date_trunc('microsec', date_trunc('hour', ct.timestamp_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', ct.timestamp_utc) / 5.0)) AS grouping
             FROM    converted_timestamps AS ct
        )

        SELECT channel_id,
               grouping AS timestamp,
               sum(quantity) AS new_subs
        FROM recent_new_subs
        WHERE grouping >= $3
        	AND grouping < $4
        GROUP BY channel_id, grouping
    `

	startDate := start.Truncate(day).AddDate(0, 0, -1).Format(SQLTimeFormat)
	endDate := end.Truncate(day).AddDate(0, 0, 2).Format(SQLTimeFormat)
	clientStartTime := start.Format(SQLTimeFormat)
	clientEndTime := end.Format(SQLTimeFormat)

	var results []SubscriptionPurchaseSuccessTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, startDate, endDate, clientStartTime, clientEndTime)
	if err != nil {
		return nil, errors.Wrap(err, "redshift get unique subscription_purchase_success timeseries: failed to query new subs by channelID")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get subscription_purchase_success timeseries: failed to close rows when querying newSubs by channelID"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit SubscriptionPurchaseSuccessTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.NewSubs,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get new_subs timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
