package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayClipsCreateAggregate queries Redshift to retrieves clip create_clip referrer breakdowns
func (c *Client) GetVideoPlayClipsCreateAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayClipsCreateAggregates, error) {
	statement := `
       WITH recent_sessions AS
        (    SELECT  *
             FROM    analysis.sessions
             WHERE   segment_start_time >= $1
             AND     segment_start_time < $2
        ), new_clips AS
        (    SELECT  channel_id,
                     date_trunc('minute', time_utc) AS time_utc
             FROM    tahoe_recent.create_clip
             WHERE   date >= $3
             AND     date < $4
        )

        SELECT a.channel_id,
               a.segment_start_time,
               count(b.channel_id) AS clip_creates
        FROM recent_sessions a
        INNER JOIN new_clips b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        GROUP BY a.channel_id, a.segment_start_time
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayClipsCreateAggregates
	rows, err := c.QueryContext(ctx, statement, start.Format(SQLTimeFormat), end.Format(SQLTimeFormat), aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get clips create_clip aggregate: query failed")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get clips create_clip aggregate: failed to close rows"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var agg VideoPlayClipsCreateAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.Creates,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get clips create_clip aggregate: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
