package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetVideoPlayClipsCreateTimeseries queries Redshift to retrieves clip create_clip referrer breakdowns
func (c *Client) GetVideoPlayClipsCreateTimeseries(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayClipsCreatesTimeseriesUnit, error) {
	statement := `
        WITH new_clips AS
        (    SELECT  channel_id,
                     date_trunc('microsec', date_trunc('hour', time_utc) + INTERVAL '5 min' * FLOOR(date_part('minute', time_utc) / 5.0)) AS grouping
             FROM    tahoe_recent.create_clip
             WHERE   date >= $1
             AND     date < $2
        )

        SELECT channel_id as channel_id,
               grouping AS timestamp,
               count(*) AS clip_creates
        FROM new_clips
        WHERE grouping >= $3
            AND grouping < $4
        GROUP BY channel_id, grouping
    `

	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	var results []VideoPlayClipsCreatesTimeseriesUnit
	rows, err := c.QueryContext(ctx, statement, aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, errors.Wrap(err, "redshift get clips create_clip timeseries: query failed")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get clips create_clip timeseries: failed to close rows"

			log.WithError(err).Error(msg)
			return
		}
	}()

	for rows.Next() {
		var unit VideoPlayClipsCreatesTimeseriesUnit
		err = rows.Scan(
			&unit.ChannelID,
			&unit.Timestamp,
			&unit.Creates,
		)

		results = append(results, unit)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get clips create_clip timeseries: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	return results, nil
}
