package redshift

import (
	"context"
	"time"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	clipsReferralTemp = `
    CREATE TEMP TABLE temp_recent_clips
    DISTSTYLE ALL
    SORTKEY (2, 3)
    AS  SELECT b.clip_id,
        a.channel_id,
        a.segment_start_time
        FROM    analysis.sessions a
        INNER JOIN tahoe_recent.create_clip b
        ON a.segment_start_time <= b.time_utc
            AND a.segment_end_time >= b.time_utc
            AND a.channel_id = b.channel_id
        WHERE   a.segment_start_time >= $3
        AND     a.segment_start_time < $4
        AND     b.date >= $1
        AND     b.date < $2
    `

	clipsReferralQuery = `
        WITH modified_clips AS
        (   SELECT DISTINCT vod_id
                FROM tahoe_recent."video-play"
            WHERE   date >= $1
            AND     date < $2
            AND     time_utc >= $3
            AND     time_utc < $4
            AND     vod_type = 'clip'
            AND     vod_id IS NOT NULL
            AND     vod_id SIMILAR TO '[0-9]%'
        ), modified_groupings AS
        (
            SELECT DISTINCT b.channel_id,
            b.segment_start_time
            FROM modified_clips a
            INNER JOIN temp_recent_clips b
            ON a.vod_id = b.clip_id
        ), grouping_referrers AS
        (
            SELECT b.clip_id,
                a.segment_start_time,
                c.referrer,
                sum(c.view_count) as views,
                a.channel_id
            FROM modified_groupings a
            INNER JOIN temp_recent_clips b
            ON a.channel_id = b.channel_id
                AND a.segment_start_time = b.segment_start_time
            INNER JOIN analysis.clips c
            ON c.clip_id = b.clip_id
            GROUP BY a.channel_id, a.segment_start_time, b.clip_id, c.referrer
        )

        SELECT channel_id,
            segment_start_time,
            referrer,
            sum(views) AS views
        FROM grouping_referrers
        GROUP BY channel_id, segment_start_time, referrer
        ORDER BY channel_id ASC, segment_start_time DESC, referrer ASC
    `

	clipsReferralCleanup = `
    DROP TABLE IF EXISTS temp_recent_clips
    `
)

// GetVideoPlayClipsReferrerAggregate queries Redshift to retrieves clip video-play referrer breakdowns
func (c *Client) GetVideoPlayClipsReferrerAggregate(ctx context.Context, start time.Time, end time.Time) ([]VideoPlayClipsReferrersAggregates, error) {
	// date in ace is in PST we should query additional data to make sure there arent gaps
	aceStart := start.Truncate(day).AddDate(0, 0, -1)
	aceEnd := end.Truncate(day).AddDate(0, 0, 2)

	// We will only recalculate clip views if it was created within the last 7 days
	aceClipCutoff := aceStart.Truncate(day).AddDate(0, 0, -7)
	startClipCutoff := start.Truncate(day).AddDate(0, 0, -7)

	var results []VideoPlayClipsReferrersAggregates
	txn, err := c.BeginTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "redshift get clips video-play aggregate: failed to start tranasction")
	}

	defer func() {
		if err != nil {
			err = txn.Rollback()

		}
		err = txn.Commit()
	}()

	_, err = txn.ExecContext(ctx, clipsReferralTemp, aceClipCutoff.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), startClipCutoff.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, err
	}

	rows, err := txn.QueryContext(ctx, clipsReferralQuery, aceStart.Format(SQLTimeFormat), aceEnd.Format(SQLTimeFormat), start.Format(SQLTimeFormat), end.Format(SQLTimeFormat))
	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "redshift get clips video-play aggregate: failed to close rows"

			log.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		var agg VideoPlayClipsReferrersAggregates
		err = rows.Scan(
			&agg.ChannelID,
			&agg.SegmentStartTime,
			&agg.Referrer,
			&agg.Views,
		)

		results = append(results, agg)
	}

	err = rows.Err()
	if err != nil {
		msg := "redshift get clips video-play aggregate: error scanning rows"

		log.WithError(err).Error(msg)
		return nil, errors.Wrap(err, msg)
	}

	// this is to handle http://go-database-sql.org/surprises.html#multiple-statement-support
	err = rows.Close()
	if err != nil {
		msg := "redshift get clips video-play aggregate: failed to close rows"

		log.WithError(err).Error(msg)
	}

	_, err = txn.ExecContext(ctx, clipsReferralCleanup)
	if err != nil {
		return nil, err
	}

	return results, nil
}
